// Database name
const DB_NAME = 'offline-uploads-db';
const STORE_NAME = 'uploads';
const DB_VERSION = 1;

// Function to completely reset and recreate the database
async function resetDatabase() {
    console.log('Complete database reset...');
    
    // Delete existing database if it exists
    return new Promise((resolve, reject) => {
        const deleteRequest = indexedDB.deleteDatabase(DB_NAME);
        
        deleteRequest.onsuccess = () => {
            console.log(`Database ${DB_NAME} successfully deleted`);
            
            // Create database from scratch
            const openRequest = indexedDB.open(DB_NAME, DB_VERSION);
            
            openRequest.onupgradeneeded = (event) => {
                console.log(`Creating new store ${STORE_NAME} during upgrade to version ${DB_VERSION}`);
                const db = event.target.result;
                db.createObjectStore(STORE_NAME, { keyPath: 'id', autoIncrement: true });
            };
            
            openRequest.onsuccess = (event) => {
                const db = event.target.result;
                console.log(`Database recreated, store ${STORE_NAME} created`);
                db.close();
                resolve();
            };
            
            openRequest.onerror = (event) => {
                console.error('Error creating database:', event.target.error);
                reject(event.target.error);
            };
        };
        
        deleteRequest.onerror = (event) => {
            console.error('Error deleting database:', event.target.error);
            reject(event.target.error);
        };
    });
}

// Check if the store exists in the database
async function checkStoreExists() {
    return new Promise((resolve, reject) => {
        const openRequest = indexedDB.open(DB_NAME);
        
        openRequest.onsuccess = (event) => {
            const db = event.target.result;
            const exists = db.objectStoreNames.contains(STORE_NAME);
            console.log(`Store ${STORE_NAME} check: ${exists ? 'exists' : 'does not exist'}`);
            db.close();
            resolve(exists);
        };
        
        openRequest.onerror = (event) => {
            console.error('Error checking store:', event.target.error);
            reject(event.target.error);
        };
    });
}

// Add a file to the store
async function addFileToStore(fileName, fileData, mimeType) {
    // First check if the store exists
    const storeExists = await checkStoreExists();
    
    // If the store doesn't exist, create it
    if (!storeExists) {
        await resetDatabase();
    }
    
    // Now add the file
    return new Promise((resolve, reject) => {
        const openRequest = indexedDB.open(DB_NAME, DB_VERSION);
        
        openRequest.onsuccess = (event) => {
            const db = event.target.result;
            
            // Double-check that the store exists
            if (!db.objectStoreNames.contains(STORE_NAME)) {
                db.close();
                console.error(`Store ${STORE_NAME} still doesn't exist after recreation!`);
                reject(new Error(`Store ${STORE_NAME} doesn't exist`));
                return;
            }
            
            try {
                const transaction = db.transaction(STORE_NAME, 'readwrite');
                const store = transaction.objectStore(STORE_NAME);
                
                const file = {
                    fileName: fileName,
                    fileData: fileData,
                    mimeType: mimeType,
                    createdAt: Date.now()
                };
                
                const request = store.add(file);
                
                request.onsuccess = () => {
                    console.log(`File successfully added to store with ID: ${request.result}`);
                    db.close();
                    resolve(request.result);
                };
                
                request.onerror = (event) => {
                    console.error('Error adding file:', event.target.error);
                    db.close();
                    reject(event.target.error);
                };
            } catch (error) {
                console.error('Error executing transaction:', error);
                db.close();
                reject(error);
            }
        };
        
        openRequest.onerror = (event) => {
            console.error('Error opening database:', event.target.error);
            reject(event.target.error);
        };
    });
}

// Get all files from the store
async function getAllFiles() {
    // First check if the store exists
    const storeExists = await checkStoreExists();
    
    // If the store doesn't exist, create it
    if (!storeExists) {
        await resetDatabase();
        return []; // Return empty array as the new store has no files
    }
    
    // Now get all files
    return new Promise((resolve, reject) => {
        const openRequest = indexedDB.open(DB_NAME, DB_VERSION);
        
        openRequest.onsuccess = (event) => {
            const db = event.target.result;
            
            // Double-check that the store exists
            if (!db.objectStoreNames.contains(STORE_NAME)) {
                db.close();
                console.error(`Store ${STORE_NAME} still doesn't exist after recreation!`);
                resolve([]); // Return empty array
                return;
            }
            
            try {
                const transaction = db.transaction(STORE_NAME, 'readonly');
                const store = transaction.objectStore(STORE_NAME);
                const request = store.getAll();
                
                request.onsuccess = () => {
                    const files = request.result;
                    console.log(`Retrieved ${files.length} files from store`);
                    db.close();
                    resolve(files);
                };
                
                request.onerror = (event) => {
                    console.error('Error retrieving files:', event.target.error);
                    db.close();
                    reject(event.target.error);
                };
            } catch (error) {
                console.error('Error executing transaction:', error);
                db.close();
                reject(error);
            }
        };
        
        openRequest.onerror = (event) => {
            console.error('Error opening database:', event.target.error);
            reject(event.target.error);
        };
    });
}

// Delete a file from the store
async function deleteFile(fileId) {
    // First check if the store exists
    const storeExists = await checkStoreExists();
    
    // If the store doesn't exist, nothing to delete
    if (!storeExists) {
        console.warn(`Store ${STORE_NAME} doesn't exist, cannot delete file`);
        return false;
    }
    
    // Now delete the file
    return new Promise((resolve, reject) => {
        const openRequest = indexedDB.open(DB_NAME, DB_VERSION);
        
        openRequest.onsuccess = (event) => {
            const db = event.target.result;
            
            try {
                const transaction = db.transaction(STORE_NAME, 'readwrite');
                const store = transaction.objectStore(STORE_NAME);
                const request = store.delete(fileId);
                
                request.onsuccess = () => {
                    console.log(`File with ID ${fileId} successfully deleted`);
                    db.close();
                    resolve(true);
                };
                
                request.onerror = (event) => {
                    console.error(`Error deleting file with ID ${fileId}:`, event.target.error);
                    db.close();
                    reject(event.target.error);
                };
            } catch (error) {
                console.error('Error executing transaction:', error);
                db.close();
                reject(error);
            }
        };
        
        openRequest.onerror = (event) => {
            console.error('Error opening database:', event.target.error);
            reject(event.target.error);
        };
    });
}

// Function to determine MIME type based on file extension or data
function detectMimeType(fileName, blob) {
    // Check file extension
    const extensions = {
        '.jpg': 'image/jpeg',
        '.jpeg': 'image/jpeg',
        '.png': 'image/png',
        '.gif': 'image/gif',
        '.webp': 'image/webp',
        '.bmp': 'image/bmp',
        '.svg': 'image/svg+xml',
        '.pdf': 'application/pdf',
        '.mp4': 'video/mp4',
        '.mp3': 'audio/mpeg'
    };
    
    // Check file extension
    for (const [ext, type] of Object.entries(extensions)) {
        if (fileName.toLowerCase().endsWith(ext)) {
            return type;
        }
    }
    
    // If not determined by extension and blob type is not application/json or octet-stream, use blob's type
    if (blob.type && 
        blob.type !== 'application/octet-stream' && 
        blob.type !== 'application/json') {
        return blob.type;
    }
    
    // Default type
    return 'application/octet-stream';
}

// Attempts to extract base64 image data from a JSON object
function extractImageBase64FromJson(data) {
    try {
        // If data is already an object, use it directly
        let jsonObject = data;
        
        // If data is a string, try to parse as JSON
        if (typeof data === 'string') {
            try {
                jsonObject = JSON.parse(data);
            } catch (e) {
                // If parsing fails, return original data
                return { success: false, originalData: data };
            }
        }
        
        // Check if there's a contents field with base64 data
        if (jsonObject && jsonObject.contents && typeof jsonObject.contents === 'string') {
            return {
                success: true,
                base64: jsonObject.contents,
                fileName: jsonObject.name || 'image.jpg',
                mimeType: jsonObject.type || 'image/jpeg'
            };
        }
        
        // If contents field not found, return original data
        return { success: false, originalData: data };
    } catch (error) {
        console.error('Error extracting image base64 from JSON:', error);
        return { success: false, originalData: data };
    }
}

// Convert Blob to base64
function blobToBase64(blob) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
    });
}

// Save file in database as base64 format
async function saveFileOffline(fileBlob, fileName) {
    try {
        // Determine correct MIME type
        const correctMimeType = detectMimeType(fileName, fileBlob);
        
        // If blob type is incorrect, create new blob with correct type
        let blobToSave = fileBlob;
        if (fileBlob.type !== correctMimeType) {
            blobToSave = new Blob([await fileBlob.arrayBuffer()], { type: correctMimeType });
        }
        
        // Get base64 data
        let base64Data = await blobToBase64(blobToSave);
        
        // Check if blob contains JSON with an image inside
        try {
            // Only if it's JSON
            if (fileBlob.type === 'application/json') {
                // Read blob as text
                const text = await new Response(fileBlob).text();
                // Try to extract image base64 from JSON
                const extracted = extractImageBase64FromJson(text);
                if (extracted.success) {
                    // If successfully extracted image base64, use it instead of JSON
                    base64Data = `data:${extracted.mimeType};base64,${extracted.base64}`;
                    fileName = extracted.fileName;
                }
            }
        } catch (e) {
            console.error('Error extracting image from JSON blob:', e);
            // Continue with original data in case of error
        }
        
        // Add file to store
        await addFileToStore(fileName, base64Data, correctMimeType);
        console.log('File saved in base64 format:', fileName, 'with type:', correctMimeType);
        
    } catch (error) {
        console.error('Error saving file in base64:', error);
    }
}

// Upload all saved files to server, sending base64 string
async function uploadSavedFiles(uploadUrl) {
    try {
        // Get all files
        const files = await getAllFiles();
        
        if (!files || files.length === 0) {
            console.log('No files to upload');
            return;
        }
        
        console.log(`Found ${files.length} files to upload`);
        
        // Process files one by one
        for (const file of files) {
            try {
                let fileData = file.fileData;
                let mimeType = file.mimeType;
                
                console.log(`Processing file: ${file.fileName}, mimeType: ${mimeType}`);
                
                // Check if file contains JSON with an image
                if (fileData && fileData.startsWith && fileData.startsWith('data:application/json;base64,')) {
                    try {
                        // Extract base64 part
                        const base64Part = fileData.split(',')[1];
                        // Decode to string
                        const jsonString = atob(base64Part);
                        // Check if JSON contains contents field with image
                        const extracted = extractImageBase64FromJson(jsonString);
                        
                        if (extracted.success) {
                            // If successfully extracted image base64, create data URL
                            fileData = `data:${extracted.mimeType};base64,${extracted.base64}`;
                            mimeType = extracted.mimeType;
                            console.log(`Extracted image from JSON: type ${mimeType}`);
                        }
                    } catch (e) {
                        console.error('Error processing JSON file:', e);
                    }
                }
                
                // If fileData doesn't start with data:, add prefix
                if (fileData && typeof fileData === 'string' && !fileData.startsWith('data:')) {
                    fileData = `data:${mimeType};base64,${fileData}`;
                    console.log('Added data: prefix to fileData');
                }
                
                console.log(`Uploading file: ${file.fileName} to ${uploadUrl}`);
                
                // Send base64 with data: prefix
                const response = await fetch(uploadUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        fileName: file.fileName,
                        fileData: fileData, // Complete string with data:mimeType;base64,...
                        mimeType: mimeType,
                        timestamp: Date.now()
                    })
                });
                
                if (response.ok) {
                    // If successfully sent, delete file from database
                    await deleteFile(file.id);
                    console.log(`Successfully uploaded and deleted offline file: ${file.fileName}`);
                } else {
                    console.error(`Upload failed with status ${response.status}:`, await response.text());
                }
            } catch (error) {
                console.error('Upload failed for file, will retry later:', file.fileName, error);
            }
        }
    } catch (error) {
        console.error('Error in upload process:', error);
    }
}

// Initialize database when service worker installs
self.addEventListener('install', (event) => {
    console.log('Service Worker installing');
    
    event.waitUntil(
        Promise.all([
            // Force database recreation
            resetDatabase(),
            
            // Cache basic files
            caches.open('my-site-cache-v1').then((cache) => {
                return cache.addAll([
                    '/',
                    '/sw.js'
                ]);
            })
        ])
    );
});

// Intercept all requests
self.addEventListener('fetch', (event) => {
    const { request } = event;
    console.log('Fetch event for ', request.url);
    if (request.method === 'POST') {
        event.respondWith(
            (async function() {
                try {
                    return await fetch(request.clone());
                } catch (error) {
                    const body = await request.clone().blob();
                    
                    // Check content if it's JSON with an image
                    let fileName = `offline-file-${Date.now()}`;
                    
                    // If it's JSON, check for image inside
                    if (body.type === 'application/json') {
                        try {
                            const text = await body.text();
                            const extracted = extractImageBase64FromJson(text);
                            if (extracted.success) {
                                fileName = extracted.fileName;
                            }
                        } catch (e) {
                            console.error('Error extracting file info from JSON:', e);
                        }
                    }
                    
                    // Add extension based on content type if possible
                    if (!fileName.includes('.')) {
                        let extension = '.bin';
                        // Check content type and give appropriate extension
                        if (body.type.startsWith('image/')) {
                            if (body.type === 'image/jpeg') extension = '.jpg';
                            else if (body.type === 'image/png') extension = '.png';
                            else if (body.type === 'image/gif') extension = '.gif';
                            else if (body.type === 'image/webp') extension = '.webp';
                            else extension = '.img';
                        }
                        fileName += extension;
                    }
                    
                    await saveFileOffline(body, fileName);
                    console.log('File saved offline:', fileName);
                    return new Response(JSON.stringify({ success: false, offline: true }), {
                        headers: { 'Content-Type': 'application/json' }
                    });
                }
            })()
        );
    } else {
        event.respondWith(
            fetch(request).catch(() => caches.match(request))
        );
    }
});

// Background Sync for uploading files
self.addEventListener('sync', (event) => {
    console.log('Background Sync event:', event.tag);
    if (event.tag === 'sync-offline-uploads') {
        event.waitUntil(
            uploadSavedFiles('https://dariamoiseenko.bubbleapps.io/api/1.1/wf/upload_offline_file') // URL for uploading files
        );
    }
});

// On activation, recreate database if it doesn't exist
self.addEventListener('activate', (event) => {
    console.log('Service Worker activating');
    
    event.waitUntil(
        checkStoreExists().then(exists => {
            if (!exists) {
                console.log('Store does not exist on activation, recreating...');
                return resetDatabase();
            }
            console.log('Store exists, doing nothing');
            return Promise.resolve();
        })
    );
});