<?php

namespace WFCO\HubSpot;

#[\AllowDynamicProperties]
abstract class WFCO_Hubspot_Call extends \WFCO_Call {
	/**
	 *
	 * @param array $required_fields
	 */
	public function __construct( $required_fields = array( 'api_key' ) ) {
		$this->required_fields = $required_fields;
	}

	/** Abstract functions that must be present in child's call class */
	abstract function process_hubspot_call();

	abstract function get_endpoint( $endpoint_var = '' );

	/** Required fields handling is done here, Also process_hubspot_call must be implemented in child call class */
	public function process() {
		$is_required_fields_present = $this->check_fields( $this->data, $this->required_fields );
		if ( false === $is_required_fields_present ) {
			return $this->get_autonami_error( $this->show_fields_error()['body'][0] );
		}

		\BWFCO_Hubspot::set_headers( $this->data['api_key'] );

		return $this->process_hubspot_call();
	}

	/**
	 * Use do_hubspot_call instead of make_wp_requests,
	 * to make use of handling response and errors from remote API call.
	 *
	 * @param array $params
	 * @param int $method
	 * @param string $endpoint_var
	 *
	 * @return array
	 */
	public function do_hubspot_call( $params, $method, $endpoint_var = '' ) {
		$response = $this->make_wp_requests( $this->get_endpoint( $endpoint_var ), $params, \BWFCO_Hubspot::get_headers(), $method );

		return $this->handle_api_response( $response );
	}

	public function get_autonami_error( $message = false ) {
		BWFAN_Core()->logger->log( $message, 'failed-' . $this->get_slug() . '-action' );

		return array(
			'status'  => 4,
			'message' => ( false !== $message ) ? $message : __( 'Unknown FunnelKit Automations Error', 'autonami-automations-connectors' )
		);
	}

	public function get_autonami_success( $message = false ) {
		return array(
			'status'  => 3,
			'message' => ( false !== $message ) ? $message : __( 'Task executed successfully!', 'autonami-automations-connectors' )
		);
	}

	/**
	 * Handle API or Autonami Response or Error
	 *
	 * @param array $res
	 *
	 * @return array
	 */
	public function handle_api_response( $res ) {
		/** If success (within 200 status), then return payload (actual response) and status, message */

		if ( ( absint( $res['response'] ) - 200 ) < 100 ) {
			return array(
				'status'  => 3,
				'payload' => $res['body'],
				'message' => __( 'Hubspot API call executed successfully', 'autonami-automations-connectors' )
			);
		}

		/** checking if the contact already exist in system */
		if ( isset( $res['response'] ) && 409 === absint( $res['response'] ) ) {
			return array(
				'status'  => 3,
				'payload' => $res['body'],
				'message' => isset( $res['body']['message'] ) ? $res['body']['message'] : __( 'Hubspot API call executed successfully', 'autonami-automations-connectors' ),
			);
		}

		/** If failed, send appropriate error */
		$response_code = __( '. Error Response Code: ', 'autonami-automations-connectors' ) . $res['response'];
		$hubspot_error = isset( $res['body']['error']['message'] ) ? __( 'Hubspot Error: ', 'autonami-automations-connectors' ) . $res['body']['error']['message'] . ', Code: ' . $res['body']['error']['code'] : '';
		$hubspot_error = isset( $res['body']['message'] ) ? __( 'Hubspot Error: ', 'autonami-automations-connectors' ) . $res['body']['message'] . ', Code: ' . $res['response'] : $hubspot_error;
		$hubspot_error = is_array( $res['body'] ) && isset( $res['body']['error'] ) ? $hubspot_error : false;
		$unknown_error = __( 'Unknown Hubspot Error', 'autonami-automations-connectors' );
		if ( ! empty( $res['body']['status'] ) && 'error' == $res['body']['status'] ) {
			$hubspot_error = $res['body']['message'];
		}

		if ( 403 === absint( $res['response'] ) ) {
			return array(
				'status'  => 3,
				'code'    => $res['response'],
				'payload' => $res['body'],
				'message' => ( false !== $hubspot_error ? $hubspot_error : $unknown_error ) . $response_code
			);
		}

		return array(
			'status'  => 4,
			'message' => ( false !== $hubspot_error ? $hubspot_error : $unknown_error ) . $response_code,
		);
	}

	public function get_contact_by_email() {
		$params = [
			'api_key' => $this->data['api_key'],
			'email'   => $this->data['email']
		];

		$connectors = \WFCO_Load_Connectors::get_instance();

		/** @var Get_Contact_By_Email $call */
		$call = $connectors->get_call( 'wfco_hubspot_get_contact_by_email' );
		$call->set_data( $params );
		$result     = $call->process();
		$contact_id = 0;
		if ( ! empty( $result['payload']['results'] ) ) {
			$contact_id = $result['payload']['results'][0]['id'];
		}

		return $contact_id;
	}

	public function create_contact() {
		$params = [
			'api_key' => $this->data['api_key'],
			'email'   => $this->data['email']
		];

		$connectors = \WFCO_Load_Connectors::get_instance();

		/** @var Create_Contact $call */
		$call = $connectors->get_call( 'wfco_hubspot_create_contact' );
		$call->set_data( $params );
		$result = $call->process();

		return $result;
	}

	public function get_products_by_name( $params, $connectors ) {

		/** @var Get_Product_By_Name $call */
		$call = $connectors->get_call( 'wfco_hubspot_get_product_by_name' );
		$call->set_data( $params );
		$result  = $call->process();
		$product = [];
		if ( ! empty( $result['payload']['results'] ) ) {
			$product_id = $result['payload']['results'][0]['id'];
			$product    = [ 'id' => $product_id, 'name' => $result['payload']['results'][0]['properties']['name'] ];
		}

		return $product;
	}

	public function create_product( $params, $connectors ) {

		/** @var Create_Product $call */
		$call = $connectors->get_call( 'wfco_hubspot_create_product' );

		$call->set_data( $params );
		$result = $call->process();

		$product = [];
		if ( ! empty( $result['payload']['id'] ) ) {
			$product_id = $result['payload']['id'];
			$product    = [ 'id' => $product_id, 'name' => $result['payload']['properties']['name'] ];
		}

		return $product;
	}

	public function create_line_item( $connectors, $product_id, $qty ) {
		/** @var Create_Line_Item $call */
		$call = $connectors->get_call( 'wfco_hubspot_create_line_item' );
		$call->set_data( [
			'api_key'    => $this->data['api_key'],
			'product_id' => $product_id,
			'qty'        => $qty
		] );
		$result       = $call->process();
		$line_item_id = 0;
		if ( ! empty( $result['payload']['id'] ) ) {
			$line_item_id = $result['payload']['id'];
		}

		return $line_item_id;
	}

	public function associate_with_deal( $params ) {
		$connectors = \WFCO_Load_Connectors::get_instance();

		/** @var Associate_Line_Item_With_Deal $call */
		$call = $connectors->get_call( 'wfco_hubspot_associate_line_item_with_deal' );
		$call->set_data( $params );

		$result = $call->process();

		return $result;
	}
}