<?php

#[AllowDynamicProperties]
class BWFAN_Connectors_Common {

	/**
	 * Initialize midnight cron
	 */
	public static function init() {
		register_deactivation_hook( WFCO_AUTONAMI_CONNECTORS_PLUGIN_DIR, array( __CLASS__, 'deactivation' ) );
		add_action( 'bwfan_run_midnight_connectors_sync', array( __CLASS__, 'sync_connectors' ) );

		/** Add Mautic Page for OAuth code */
		add_action( 'admin_menu', array( __CLASS__, 'register_mautic_connector_auth_page' ) );
	}

	/**
	 * Register admin menu for Mautic
	 */
	public static function register_mautic_connector_auth_page() {
		add_submenu_page( '', 'FunnelKit Automations Mautic Auth', 'FunnelKit Automations Mautic Auth', 'manage_options', 'bwfan-connector-mautic-auth', array( __CLASS__, 'bwfan_connector_mautic_auth_callback' ) );
	}

	/**
	 * Admin page: Mautic view
	 */
	public static function bwfan_connector_mautic_auth_callback() {
		$status  = 'failed';
		$message = '';
		if ( isset( $_GET['code'] ) && isset( $_GET['state'] ) ) {

			/** @var BWFCO_Mautic $mautic */
			$mautic = WFCO_Load_Connectors::get_connector( 'bwfco_mautic' );
			$result = $mautic->exchange_code_for_tokens( $_GET['code'], $_GET['state'] );
			if ( ! is_wp_error( $result ) ) {
				$response = $mautic->handle_settings_form( $result, 'save' );
				if ( isset( $response['status'] ) ) {
					$status  = ( 'success' === $response['status'] ) ? 'success' : 'failed';
					$message = ( 'failed' === $response['status'] ) ? $response['message'] : '';
				}
			} else {
				$message = $result->get_error_message();
			}
		}

		$args = array(
			'page'           => 'autonami',
			'path'           => rawurlencode( '/connectors' ),
			'wfco_connector' => 'bwfco_mautic',
			'status'         => $status
		);
		if ( 'failed' === $status ) {
			$args['error_msg'] = $message;
		}

		wp_safe_redirect( add_query_arg( $args, site_url() . '/wp-admin/admin.php' ) );
	}

	/**
	 * Midnight cron callback
	 */
	public static function sync_connectors() {
		BWFAN_Core()->logger->log( __( 'Midnight Sync START' ), 'connector_midnight_sync' );
		$active_connectors = WFCO_Load_Connectors::get_active_connectors();
		if ( empty( $active_connectors ) || ! is_array( $active_connectors ) ) {
			BWFAN_Core()->logger->log( __( 'No Active Connectors, Midnight Sync END' ), 'connector_midnight_sync' );

			return;
		}
		foreach ( $active_connectors as $connector ) {
			$connector_slug     = $connector->get_slug();
			$connector_settings = self::get_connector_settings( $connector_slug );
			if ( false === $connector->is_syncable() || false === $connector_settings || ! isset( $connector_settings['id'] ) ) {
				continue;
			}
			$connector_id = array( 'id' => $connector_settings['id'] );
			$response     = self::sync_the_connector( $connector, $connector_id );
			$data_changed = isset( $response['data_changed'] ) ? $response['data_changed'] : 0;

			BWFAN_Core()->logger->log( __( 'Sync Report for ' . $connector_slug . ':' ), 'connector_midnight_sync' );
			if ( isset( $response['error'] ) ) {
				BWFAN_Core()->logger->log( __( "Error: " . $response['error'] . "<br/>-------------------" ), 'connector_midnight_sync' );
				continue;
			}

			BWFAN_Core()->logger->log( __( 'Status: ' . $response['status'] . ' & Message: ' . $response['message'] ), 'connector_midnight_sync' );
			BWFAN_Core()->logger->log( __( 'Data Changed: ' . ( 1 === absint( $data_changed ) ? 'Yes' : 'No' ) ), 'connector_midnight_sync' );
			BWFAN_Core()->logger->log( __( '-------------------' ), 'connector_midnight_sync' );
		}
		BWFAN_Core()->logger->log( __( 'Midnight Sync END' ), 'connector_midnight_sync' );
	}

	/**
	 * @param $connector
	 * @param $ids
	 *
	 * @return array
	 */
	public static function sync_the_connector( $connector, $ids ) {
		try {
			$response = $connector->handle_settings_form( $ids, 'sync' );
		} catch ( Exception $exception ) {
			return array( 'error' => $exception->getMessage() );
		}

		return $response;
	}

	/**
	 * Get saved connector settings by slug
	 *
	 * @param $slug
	 *
	 * @return array|false
	 */
	public static function get_connector_settings( $slug ) {
		if ( false === WFCO_Common::$saved_data ) {
			WFCO_Common::get_connectors_data();
		}
		$data = WFCO_Common::$connectors_saved_data;

		return ( isset( $data[ $slug ] ) && is_array( $data[ $slug ] ) ) ? $data[ $slug ] : false;
	}

	/**
	 * Un-schedule action on plugin deactivation
	 */
	public static function deactivation() {
		bwf_unschedule_actions( 'bwfan_run_midnight_connectors_sync' );
	}

	/**
	 * Insert SMS row in engagement table
	 * Attach UTM params if available
	 *
	 * @param $body
	 * @param $action_data
	 *
	 * @return array|mixed|string|string[]|null
	 */
	public static function modify_sms_body( $body, $action_data ) {
		/** No Pro, means no engagement tracking, UTM, Conversation features */
		if ( ! bwfan_is_autonami_pro_active() ) {
			return $body;
		}

		$body = BWFAN_Core()->conversations->insert_automation_conversation( $action_data, $body, BWFAN_Email_Conversations::$MODE_SMS );
		$utm  = BWFAN_UTM_Tracking::get_instance();
		if ( version_compare( BWFAN_PRO_VERSION, '2.0.4', '>' ) ) {
			$body = $utm->maybe_add_utm_parameters( $body, $action_data, 'sms' );
		} else {
			$body = $utm->maybe_add_utm_parameters( $body, $action_data );
		}

		return $body;
	}

	/**
	 * @param $string
	 *
	 * this function returns shorter url conditionally
	 *
	 * @return string
	 */
	public static function get_shorten_url( $string ) {

		/**
		 * If method does not exist then simply return the string shorten by Bitly
		 */
		if ( ! method_exists( 'BWFAN_Common', 'get_default_shortener_provider' ) ) {
			return self::get_parsed_string( $string );
		}

		$service = BWFAN_Common::get_default_shortener_provider();
		/**
		 * if class or method does not exist then simply return the string shorten by Bitly
		 */
		if ( empty( $service ) || ! class_exists( $service ) || ! method_exists( $service, 'get_instance' ) ) {
			return self::get_parsed_string( $string );
		}

		$ins = $service::get_instance();
		if ( ! empty( $ins ) && method_exists( $ins, 'generate_shorten_url' ) ) {
			return $ins->generate_shorten_url( $string );
		}

		return self::get_parsed_string( $string );
	}

	public static function get_parsed_string( $string ) {
		if ( shortcode_exists( 'bwfan_bitly_shorten' ) ) {
			return do_shortcode( '[bwfan_bitly_shorten]' . $string . '[/bwfan_bitly_shorten]' );
		}

		return $string;
	}

	/**
	 * Get Contact id from contact object
	 * Available in Pro plugin as well
	 * But complete code here
	 *
	 * @param $email
	 * @param $user_id
	 * @param $phone
	 *
	 * @return int
	 */
	public static function get_cid_from_contact( $email = '', $user_id = 0, $phone = '' ) {
		if ( empty( $email ) && empty( $user_id ) && empty( $phone ) ) {
			return 0;
		}

		$contact = new WooFunnels_Contact( $user_id, $email, $phone );
		if ( $contact->get_id() > 0 ) {
			return $contact->get_id();
		}

		if ( empty( $phone ) ) {
			return 0;
		}

		$new_phone = self::get_phone_without_code( $phone );
		if ( $new_phone === $phone ) {
			return 0;
		}

		$contact = new WooFunnels_Contact( '', '', $new_phone );
		if ( $contact->get_id() > 0 ) {
			return $contact->get_id();
		}

		return 0;
	}

	public static function get_phone_without_code( $phone ) {
		if ( ! BWFAN_Core()->connectors->is_connected( 'bwfco_twilio' ) ) {
			return $phone;
		}

		WFCO_Common::get_connectors_data();
		$settings = WFCO_Common::$connectors_saved_data['bwfco_twilio'];
		if ( empty( $settings ) ) {
			return $phone;
		}
		$connectors = WFCO_Load_Connectors::get_instance();
		$call       = $connectors->get_call( 'wfco_twilio_get_country_code' );
		$call->set_data( array(
			'account_sid' => isset( $settings['account_sid'] ) ? $settings['account_sid'] : '',
			'auth_token'  => isset( $settings['auth_token'] ) ? $settings['auth_token'] : '',
			'phone'       => $phone
		) );
		$result = $call->process();

		if ( isset( $result['response'] ) && 200 === $result['response'] ) {
			$country_code = isset( $result['body'] ) && isset( $result['body']['country_code'] ) ? $result['body']['country_code'] : '';
		}
		$phone_code = BWFAN_Phone_Numbers::get_area_code( $country_code );

		return preg_replace( '/^\+?' . $phone_code . '|\|1|\D/', '', ( $phone ) );
	}

	/**
	 * Check if string contains unicode character
	 *
	 * @param $message
	 *
	 * @return bool
	 */
	public static function is_unicode( $message ) {
		return ( ! empty( $message ) && ( strlen( $message ) !== strlen( utf8_decode( $message ) ) ) );
	}
}
