<?php
#[AllowDynamicProperties]
final class BWFAN_AC_Create_Deal_Note extends BWFAN_Action {

	private static $instance = null;

	private function __construct() {
		$this->action_name     = __( 'Create Deal Note', 'autonami-automations-connectors' );
		$this->action_desc     = __( 'This action creates a note against a deal. Note: Deal should be created first', 'autonami-automations-connectors' );
		$this->included_events = array(
			'wc_new_order',
			'wc_order_note_added',
			'wc_order_status_change',
			'wc_product_purchased',
			'wc_product_refunded',
			'wc_product_stock_reduced',
		);
		$this->action_priority = 55;
		$this->support_v2      = true;
	}

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function load_hooks() {
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_assets' ), 98 );
	}

	/**
	 * Localize data for html fields for the current action.
	 */
	public function admin_enqueue_assets() {
		if ( BWFAN_Common::is_load_admin_assets( 'automation' ) ) {
			$data = $this->get_view_data();
			BWFAN_Core()->admin->set_actions_js_data( $this->get_class_slug(), 'pipeline_options', $data['pipelines'] );
		}
	}

	public function get_view_data() {
		$pipelines = WFCO_Common::get_single_connector_data( $this->connector, 'pipelines' );

		return array(
			'pipelines' => $pipelines,
		);
	}

	/**
	 * Show the html fields for the current action.
	 */
	public function get_view() {
		$unique_slug = $this->get_slug();
		?>
        <script type="text/html" id="tmpl-action-<?php echo esc_attr__( $unique_slug ); ?>">
            <#
            selected_pipeline_id = (_.has(data.actionSavedData, 'data') && _.has(data.actionSavedData.data, 'pipeline_id')) ? data.actionSavedData.data.pipeline_id : '';
            entered_deal_note = (_.has(data.actionSavedData, 'data') && _.has(data.actionSavedData.data, 'deal_note')) ? data.actionSavedData.data.deal_note : '';
            pipeline_options = data.actionFieldsOptions.pipeline_options;
            #>

            <div class="bwfan-input-form clearfix">
                <label for="" class="bwfan-label-title"><?php echo esc_html__( 'Select Deal Group', 'autonami-automations-connectors' ); ?></label>
                <select required id="" class="bwfan-input-wrapper" name="bwfan[{{data.action_id}}][data][pipeline_id]">
                    <option value=""><?php echo esc_html__( 'Select Group', 'autonami-automations-connectors' ); ?></option>
                    <# _.each( pipeline_options, function( value, key ){
                    selected = (selected_pipeline_id == key) ? 'selected' : '';
                    #>
                    <option value="{{key}}" {{selected}}>{{value}}</option>
                    <# }) #>

                </select>
            </div>
            <div class="bwfan-input-form clearfix">
                <label for="" class="bwfan-label-title">
					<?php
					echo esc_html__( 'Deal Description', 'autonami-automations-connectors' );
					$message = __( 'Add description of the note to be created.', 'autonami-automations-connectors' );
					echo $this->add_description( $message, '2xl', 'right' ); //phpcs:ignore WordPress.Security.EscapeOutput
					echo $this->inline_merge_tag_invoke(); //phpcs:ignore WordPress.Security.EscapeOutput
					?>
                </label>
                <textarea required class="bwfan-input-wrapper" rows="4" placeholder="<?php echo esc_attr__( 'Message', 'autonami-automations-connectors' ); ?>" name="bwfan[{{data.action_id}}][data][deal_note]">{{entered_deal_note}}</textarea>
            </div>
        </script>
		<?php
	}

	/**
	 * Make all the data which is required by the current action.
	 * This data will be used while executing the task of this action.
	 *
	 * @param $integration_object BWFAN_Integration
	 * @param $task_meta
	 *
	 * @return array|void
	 */
	public function make_data( $integration_object, $task_meta ) {
		$data_to_set                = array();
		$data_to_set['api_key']     = $integration_object->get_settings( 'api_key' );
		$data_to_set['api_url']     = $integration_object->get_settings( 'api_url' );
		$data_to_set['email']       = $task_meta['global']['email'];
		$data_to_set['deal_note']   = ( ! empty( $task_meta['data']['deal_note'] ) ) ? BWFAN_Common::decode_merge_tags( $task_meta['data']['deal_note'] ) : '';
		$data_to_set['pipeline_id'] = $task_meta['data']['pipeline_id'];

		return $data_to_set;
	}

	public function make_v2_data( $automation_data, $step_data ) {
		$data_to_set                = array();
		$data_to_set['api_key']     = isset( $step_data['connector_data']['api_key'] ) ? $step_data['connector_data']['api_key'] : '';
		$data_to_set['api_url']     = isset( $step_data['connector_data']['api_url'] ) ? $step_data['connector_data']['api_url'] : '';
		$data_to_set['email']       = $automation_data['global']['email'];
		$data_to_set['pipeline_id'] = $step_data['pipeline_id'];

		$order_id = $automation_data['global']['order_id'];
		$order    = wc_get_order( $order_id );
		$deal_id  = $order instanceof WC_Order ? $order->get_meta( '_bwfan_ac_deal_id' ) : '';

		if ( empty( $deal_id ) ) { // then get the deal id using search call
			$connectors       = WFCO_Load_Connectors::get_instance();
			$deal_search_call = $connectors->get_call( 'wfco_ac_get_deal_by_email' );

			$deal_search_call->set_data( $data_to_set );
			$response = $deal_search_call->process();
			if ( is_array( $response ) && isset( $response['response'] ) && 200 === absint( $response['response'] ) && is_array( $response['body']['deals'] ) && count( $response['body']['deals'] ) > 0 ) {
				$deal_id = $response['body']['deals'][0]['id'];
			}
		}
		$data_to_set['deal_note'] = ( ! empty( $step_data['deal_note'] ) ) ? BWFAN_Common::decode_merge_tags( $step_data['deal_note'] ) : '';
		$data_to_set['deal_id']   = $deal_id;

		return $data_to_set;
	}

	/**
	 * Execute the current action.
	 * Return 3 for successful execution , 4 for permanent failure.
	 *
	 * @param $action_data
	 *
	 * @return array
	 */
	public function execute_action( $action_data ) {
		$deal_id = 0;
		if ( isset( $action_data['processed_data']['order_id'] ) ) {
			$order_id = $action_data['processed_data']['order_id'];
			$order    = wc_get_order( $order_id );
			$deal_id  = $order instanceof WC_Order ? $order->get_meta( '_bwfan_ac_deal_id' ) : 0;
			$deal_id  = intval( $deal_id );
		}
		// dynamic filter, it will be like 'bwfan_get_deal_id_wc' or 'bwfan_get_deal_id_edd'. Created to get deal id from order meta.
		$deal_id = apply_filters( 'bwfan_get_deal_id_' . $action_data['event_data']['event_source'], $deal_id, $action_data );

		if ( empty( $deal_id ) ) { // then get the deal id using search call
			$connectors       = WFCO_Load_Connectors::get_instance();
			$deal_search_call = $connectors->get_call( 'wfco_ac_get_deal_by_email' );
			$data             = array(
				'api_key'     => $action_data['processed_data']['api_key'],
				'api_url'     => $action_data['processed_data']['api_url'],
				'email'       => $action_data['processed_data']['email'],
				'pipeline_id' => $action_data['processed_data']['pipeline_id'],
			);
			$deal_search_call->set_data( $data );
			$response = $deal_search_call->process();
			if ( is_array( $response ) && isset( $response['response'] ) && 200 === absint( $response['response'] ) && is_array( $response['body']['deals'] ) && count( $response['body']['deals'] ) > 0 ) {
				$deal_id = $response['body']['deals'][0]['id'];
			}
		}

		if ( empty( $deal_id ) ) { // still empty after search than return
			$result                         = [];
			$result['bwfan_custom_message'] = __( 'Deal Not Created', 'autonami-automations-connectors' );

			return $result;
		}

		$action_data['processed_data']['deal_id'] = $deal_id;

		return parent::execute_action( $action_data );
	}

	protected function handle_response( $result, $call_object = null ) {
		if ( isset( $result['status'] ) ) {
			return $result;
		}

		if ( isset( $result['body']['note'] ) && is_array( $result['body']['note'] ) && count( $result['body']['note'] ) > 0 ) {
			return array(
				'status'  => 3,
				'message' => isset( $result['body']['result_message'] ) ? $result['body']['result_message'] : __( 'Deal note successfully created', 'autonami-automations-connectors' ),
			);
		}

		if ( 502 === absint( $result['response'] ) && is_array( $result['body'] ) ) {
			return array(
				'status'  => 4,
				'message' => isset( $result['body'][0] ) ? $result['body'][0] : __( 'Unknown FunnelKit Automations Error', 'autonami-automations-connectors' ),
			);
		}

		$response_code   = __( '. Response Code: ', 'autonami-automations-connectors' ) . $result['response'];
		$result_message  = ( is_array( $result['body'] ) && isset( $result['body']['result_message'] ) ) ? $result['body']['result_message'] : false;
		$message         = ( is_array( $result['body'] ) && isset( $result['body']['message'] ) ) ? $result['body']['message'] : false;
		$unknown_message = __( 'Unknown API Exception', 'autonami-automations-connectors' );

		return array(
			'status'  => 4,
			'message' => ( false !== $result_message ? $result_message : ( false !== $message ? $message : $unknown_message ) ) . $response_code,
		);
	}

	public function handle_response_v2( $result ) {
		$message = __( 'Unknown Error', 'autonami-automations-connectors' );
		if ( ! isset( $result['response'] ) ) {
			return $this->error_response( $message );
		}

		if ( 502 === absint( $result['response'] ) && is_array( $result['body'] ) ) {
			$message = isset( $result['body'][0] ) ? $result['body'][0] : __( 'Unknown FunnelKit Automations Error', 'autonami-automations-connectors' );
		}

		if ( is_array( $result['body'] ) && isset( $result['body']['errors'] ) ) {
			$message = __( 'Error: ' . $result['body']['errors'][0]['title'], 'autonami-automations-connectors' );
		}

		if ( 200 !== $result['response'] ) {
			return $this->error_response( $message );
		}

		$response_code   = __( '. Response Code: ', 'autonami-automations-connectors' ) . $result['response'];
		$result_message  = ( is_array( $result['body'] ) && isset( $result['body']['result_message'] ) ) ? $result['body']['result_message'] : false;
		$message         = ( is_array( $result['body'] ) && isset( $result['body']['message'] ) ) ? $result['body']['message'] : false;
		$unknown_message = __( 'Unknown API Exception', 'autonami-automations-connectors' );

		return $this->success_message( ( false !== $result_message ? $result_message : ( false !== $message ? $message : $unknown_message ) ) . $response_code );
	}

	/**
	 * v2 Method: Get field Schema
	 *
	 * @return array[]
	 */
	public function get_fields_schema() {
		$data     = $this->get_view_data();
		$pipeline = isset( $data['pipelines'] ) && ! empty( $data['pipelines'] ) && is_array( $data['pipelines'] ) ? BWFAN_PRO_Common::prepared_field_options( array_replace( [ '' => 'Select' ], $data['pipelines'] ) ) : [];

		return [
			[
				'id'          => 'pipeline_id',
				'label'       => __( "Select Pipeline", 'autonami-automations-connectors' ),
				'type'        => 'select',
				'options'     => $pipeline,
				'placeholder' => "Choose Pipeline",
				"class"       => 'bwfan-input-wrapper',
				'tip'         => __( '', 'autonami-automations-connectors' ),
				"description" => '',
				"required"    => true,
			],
			[
				'id'          => 'deal_note',
				'label'       => __( "Deal Note", 'autonami-automations-connectors' ),
				'type'        => 'textarea',
				'placeholder' => "Note",
				"class"       => 'bwfan-input-wrapper',
				'tip'         => __( 'Add description of the note to be created.', 'autonami-automations-connectors' ),
				"description" => '',
				"required"    => true,
			],
		];
	}

	public function get_desc_text( $data ) {
		$data = json_decode( wp_json_encode( $data ), true );
		if ( ! isset( $data['deal_note'] ) || empty( $data['deal_note'] ) ) {
			return '';
		}

		return $data['deal_note'];
	}

}

return 'BWFAN_AC_Create_Deal_Note';
