<?php

/**
 * Class BWFAN_Twilio_Message_Received
 */
#[AllowDynamicProperties]
final class BWFAN_Twilio_Message_Received extends BWFAN_Event {
	private static $instance = null;
	private $twilio_id = null;
	private $automation_key = '';
	private $localized_automation_key = '';
	private $twilio_data = array();

	private function __construct() {
		$this->optgroup_label         = __( 'Twilio', 'autonami-automations' );
		$this->event_name             = __( 'SMS Received', 'autonami-automations' );
		$this->event_desc             = __( 'This automation would trigger webhook, from Twilio Message Received.', 'autonami-automations' );
		$this->event_merge_tag_groups = array( 'twilio_sms' );
		$this->event_rule_groups      = array( 'twilio_sms' );
		$this->included_actions       = array(
			'wp_http_post',
			'wp_debug',
			'wp_custom_callback',
			'za_send_data',
			'twilio_send_sms',
			'sl_message_user',
			'sl_message'
		);
		$this->v2                     = true;
		$this->need_unique_key        = true;
	}

	public function load_hooks() {
		add_action( 'admin_enqueue_scripts', array( $this, 'bwfan_webhook_admin_enqueue_assets' ), 98 );
		add_action( "bwfan_twilio_connector_sync_call", array( $this, 'before_process_webhook_contact' ), 10, 3 );

		add_action( 'bwfan_webhook_twilio_sms_received', array( $this, 'process' ), 10, 3 );
	}

	public function before_process_webhook_contact( $automation_id, $automation_key, $request_data ) {
		$hook  = 'bwfan_webhook_twilio_sms_received';
		$args  = array(
			'twilio_data'    => $request_data,
			'automation_key' => $automation_key,
			'automation_id'  => $automation_id
		);
		$group = 'twilio';

		if ( bwf_has_action_scheduled( $hook, $args, $group ) ) {
			return;
		}
		bwf_schedule_single_action( time(), $hook, $args, $group );
	}

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function get_twilio_phone() {
		if ( ! isset( $this->twilio_data['From'] ) || empty( $this->twilio_data['From'] ) ) {
			return false;
		}

		return $this->twilio_data['From'];
	}

	public function get_order_from_phone() {
		$phone = $this->get_twilio_phone();
		if ( false === $phone || ! bwfan_is_woocommerce_active() ) {
			return false;
		}
		$args   = array(
			'billing_phone' => $phone,
		);
		$orders = wc_get_orders( $args );
		if ( empty( $orders ) ) {
			return false;
		}

		return $orders[0];
	}

	public function get_email_event() {
		$bwf_contact = $this->get_contact_by_phone();
		if ( $bwf_contact instanceof WooFunnels_Contact && $bwf_contact->get_id() > 0 ) {
			return $bwf_contact->get_email();
		}

		$order = $this->get_order_from_phone();

		return $order instanceof WC_Order ? $order->get_billing_email() : false;
	}

	public function get_user_id_event() {
		$order = $this->get_order_from_phone();

		return $order instanceof WC_Order ? $order->get_user_id() : false;
	}

	public function get_contact_by_phone() {
		$phone = $this->get_twilio_phone();
		if ( empty( $phone ) ) {
			return false;
		}

		$cid = BWFAN_Connectors_Common::get_cid_from_contact( '', '', $phone );
		if ( empty( $cid ) ) {
			return false;
		}

		return new WooFunnels_Contact( '', '', '', $cid );
	}

	public function bwfan_webhook_admin_enqueue_assets() {
		$this->twilio_id = isset( $_GET['edit'] ) ? sanitize_text_field( $_GET['edit'] ) : ''; //phpcs:ignore WordPress.Security.NonceVerification
		if ( ! empty( $this->twilio_id ) ) {
			$meta = BWFAN_Model_Automationmeta::get_meta( $this->twilio_id, 'event_meta' );
			if ( isset( $meta['bwfan_unique_key'] ) && ! empty( $meta['bwfan_unique_key'] ) ) {
				$this->localized_automation_key = $meta['bwfan_unique_key'];
			}
		}

		BWFAN_Core()->admin->set_events_js_data( $this->get_slug(), 'automation_id', $this->twilio_id );
		BWFAN_Core()->admin->set_events_js_data( $this->get_slug(), 'saved_localized_automation_key', $this->localized_automation_key );
	}

	/**
	 * Make the view data for the current event which will be shown in task listing screen.
	 *
	 * @param $global_data
	 *
	 * @return false|string
	 */
	public function get_task_view( $global_data ) {
		ob_start();

		if ( ! is_array( $global_data['twilio_data'] ) || empty( $global_data['twilio_data'] ) ) {
			?>
            <li>
                <strong><?php esc_html_e( 'No Data Available', 'autonami-automations-connectors' ); ?> </strong>
            </li>
			<?php
			return ob_get_clean();
		}

		$data_to_show = array(
			'From'        => 'From',
			'FromCountry' => 'Country',
			'Body'        => 'Message'
		);

		foreach ( $data_to_show as $key => $alias ) {
			if ( ! isset( $global_data['twilio_data'][ $key ] ) || empty( $global_data['twilio_data'][ $key ] ) ) {
				continue;
			}
			?>
            <li>
                <strong><?php esc_html_e( "$alias: " ); ?></strong>
				<?php esc_html_e( $global_data['twilio_data'][ $key ] ); ?>
            </li>
			<?php
		}

		return ob_get_clean();
	}

	/**
	 * Show the html fields for the current event.
	 */
	public function get_view( $db_eventmeta_saved_value ) {
		?>
        <script type="text/html" id="tmpl-event-<?php esc_attr_e( $this->get_slug() ); ?>">
            <#
            var eventslug = '<?php esc_html_e( $this->get_slug() ); ?>';
            var eventData = bwfan_events_js_data[eventslug];
            var event_save_unique_key =eventData.saved_localized_automation_key;
            if(event_save_unique_key.length>0){
            eventData.localized_automation_key = event_save_unique_key
            }
            var webhook_url = '<?php esc_attr_e( home_url( '/' ) ); ?>wp-json/autonami/v1/twilio/webhook?twilio_id='+eventData.automation_id+'&twilio_key='+eventData.localized_automation_key;
            #>
            <div class="bwfan_mt15"></div>
            <label for="bwfan-webhook-url" class="bwfan-label-title"><?php esc_html_e( 'URL', 'autonami-automations-connectors' ); ?></label>
            <div class="bwfan-textarea-box">
                <textarea name="event_meta[bwfan_webhook_url]" class="bwfan-input-wrapper bwfan-webhook-url" id="bwfan-webhook-url" cols="45" rows="2" onclick="select();" readonly>{{webhook_url}}</textarea>
                <input type="hidden" name="event_meta[bwfan_unique_key]" id="bwfan-unique-key" value={{eventData.localized_automation_key}}>
            </div>
        </script>
		<?php
	}

	public function process( $twilio_data, $automation_key, $automation_id ) {
		$this->twilio_id      = $automation_id;
		$this->automation_key = $automation_key;
		$this->twilio_data    = $twilio_data;

		$contact_data_v2 = array(
			'a_id'           => $this->twilio_id,
			'automation_key' => $this->automation_key,
			'twilio_data'    => $this->twilio_data,
			'phone'          => $this->get_twilio_phone(),
			'email'          => $this->get_email_event(),
			'event'          => $this->get_slug(),
			'version'        => 2
		);
		BWFAN_Common::maybe_run_v2_automations( $this->get_slug(), $contact_data_v2 );

		return $this->run_automations();
	}

	/**
	 * Set up rules data
	 *
	 * @param $automation_data
	 */
	public function pre_executable_actions( $automation_data ) {
		BWFAN_Core()->rules->setRulesData( $this->twilio_id, 'twilio_id' );
		BWFAN_Core()->rules->setRulesData( $this->automation_key, 'automation_key' );
		BWFAN_Core()->rules->setRulesData( $this->twilio_data, 'twilio_data' );
	}

	/**
	 * A controller function to run automation every time an appropriate event occurs
	 * usually called by the event class just after the event hook to load all automations and run.
	 *
	 * @return array|bool
	 */
	public function run_automations() {
		BWFAN_Core()->public->load_active_automations( $this->get_slug() );
		if ( ! is_array( $this->automations_arr ) || count( $this->automations_arr ) === 0 ) {
			if ( $this->sync_start_time > 0 ) {
				/** Sync process */
				BWFAN_Core()->logger->log( 'Sync #' . $this->sync_id . '. No active automations found for Event ' . $this->get_slug(), 'sync' );

				return false;
			}
			BWFAN_Core()->logger->log( 'Async callback: No active automations found. Event - ' . $this->get_slug(), $this->log_type );

			return false;
		}

		$automation_actions = [];
		foreach ( $this->automations_arr as $automation_id => $automation_data ) {
			if ( $this->get_slug() !== $automation_data['event'] || 0 !== intval( $automation_data['requires_update'] ) ) {
				continue;
			}

			/** check if the automation_key match with the post data */
			$unique_key_matched    = ( isset( $automation_data['event_meta']['bwfan_unique_key'] ) && $this->automation_key === $automation_data['event_meta']['bwfan_unique_key'] );
			$automation_id_matched = ( absint( $automation_id ) === absint( $this->twilio_id ) );
			$ran_actions           = 0;
			if ( $unique_key_matched && $automation_id_matched ) {
				$ran_actions = $this->handle_single_automation_run( $automation_data, $automation_id );
			}

			$automation_actions[ $automation_id ] = $ran_actions;
		}

		return $automation_actions;
	}

	/**
	 * Registers the tasks for current event.
	 *
	 * @param $automation_id
	 * @param $integration_data
	 * @param $event_data
	 */
	public function register_tasks( $automation_id, $integration_data, $event_data ) {
		if ( ! is_array( $integration_data ) ) {
			return;
		}

		$meta = BWFAN_Model_Automationmeta::get_meta( $automation_id, 'event_meta' );
		if ( '' === $meta || ! is_array( $meta ) ) {
			return;
		}

		$data_to_send = $this->get_event_data();
		$this->create_tasks( $automation_id, $integration_data, $event_data, $data_to_send );
	}

	public function get_event_data() {
		$data_to_send                             = [];
		$data_to_send['global']['twilio_id']      = $this->twilio_id;
		$data_to_send['global']['automation_key'] = $this->automation_key;
		$data_to_send['global']['twilio_data']    = $this->twilio_data;
		$data_to_send['global']['phone']          = $this->get_twilio_phone();
		$data_to_send['global']['email']          = $this->get_email_event();

		return $data_to_send;
	}

	public function set_merge_tags_data( $task_meta ) {
		$merge_data                = [];
		$merge_data['twilio_data'] = $task_meta['global']['twilio_data'];
		$merge_data['phone']       = $task_meta['global']['phone'];
		$merge_data['email']       = $task_meta['global']['email'];
		BWFAN_Merge_Tag_Loader::set_data( $merge_data );
	}

	/**
	 * v2 Method: Get field Schema
	 *
	 * @return array[]
	 */
	public function get_fields_schema() {
		return [
			[
				'id'          => 'bwfan_twilio_webhook_url',
				'label'       => __( "URL", 'autonami-automations-connectors' ),
				'type'        => 'webhook',
				'webhook_url' => rest_url( 'autonami/v1/twilio/webhook/' ) . '?twilio_id={{automationId}}&twilio_key={{uniqueKey}}',
				'tip'         => __( '', 'autonami-automations-connectors' ),
				"description" => '',
				"required"    => false,
			],
		];
	}
}

/**
 * Register this event to a source.
 * This will show the current event in dropdown in single automation screen.
 */
$saved_connectors = WFCO_Common::$connectors_saved_data;

if ( empty( $saved_connectors ) ) {
	WFCO_Common::get_connectors_data();
	$saved_connectors = WFCO_Common::$connectors_saved_data;
}

if ( array_key_exists( 'bwfco_twilio', $saved_connectors ) ) {
	return 'BWFAN_Twilio_Message_Received';
}
