<?php

namespace BWFCRM\Actions\Autonami;

use BWFCRM\Actions\Base;
use BWFCRM\Calls\Autonami as Calls;
use WP_Error;

/**
 * Add tags action class
 */
class Add_Tags extends Base {

	/**
	 * Class constructor
	 */
	public function __construct() {
		$this->slug        = 'add_tags';
		$this->nice_name   = 'Add Tags';
		$this->group       = 'autonami';
		$this->group_label = 'FunnelKit Automations';
		$this->priority    = 10;
		$this->support     = [ 1, 2 ];
		$this->event_slug  = 'crm_assigned_tag';
	}

	/**
	 * Returns action field schema
	 *
	 * @return array
	 */
	public function get_action_schema() {
		return [
			'type' => 'search',
			'meta' => [
				'autocompleter' => 'tag',
				'addnew'        => true,
			]
		];
	}

	/**
	 *  process action
	 *
	 * @param $contact \BWFCRM_Contact
	 * @param $tags
	 *
	 * @return array|false
	 */
	public function handle_action( $contact, $tags ) {
		$contact_tags = $contact->get_tags();

		// format tag data
		$tags = array_map( function ( $tag ) {
			if ( ! empty( $tag ) && isset( $tag['name'] ) ) {
				$tag['value'] = $tag['name'];
				unset( $tag['name'] );
			}

			return $tag;
		}, $tags );

		if ( empty( $tags ) ) {
			return 'skip';
		}

		/**
		 * Check if call exists
		 */
		if ( ! class_exists( 'BWFCRM\Calls\Autonami\Add_Tags' ) ) {
			return false;
		}

		$call_obj = new Calls\Add_Tags;

		/**
		 * Process call
		 */
		$tags_added = $call_obj->process_call( $contact, $tags );

		return $this->get_response( $tags, $tags_added );
	}

	public function get_response( $tags, $tags_added ) {
		$message = array();
		if ( $tags_added instanceof WP_Error ) {
			$message[] = __( 'Tag(s) Add Failed: ', 'wp-marketing-automations-crm' ) . array_reduce( $tags, function ( $carry, $item ) {
					$name = $item['value'];
					if ( empty( $carry ) ) {
						return $name;
					}

					return ! empty( $name ) ? $carry . ', ' . $name : $carry;
				}, '' );
			$message[] = __( 'Tag(s) Add Failed. Error: ', 'wp-marketing-automations-crm' ) . $tags_added->get_error_message();

			return array(
				'status'  => self::$RESPONSE_FAILED,
				'message' => $message,
			);
		}

		if ( isset( $tags_added['skipped'] ) && count( $tags_added['skipped'] ) > 0 ) {
			$message[] = __( 'Tag(s) Add Skipped: ', 'wp-marketing-automations-crm' ) . array_reduce( $tags_added['skipped'], function ( $carry, $item ) {
					$name = $item->get_name();
					if ( empty( $carry ) ) {
						return $name;
					}

					return ! empty( $name ) ? $carry . ', ' . $name : $carry;
				}, '' );
		}

		if ( isset( $tags_added['assigned'] ) && count( $tags_added['assigned'] ) > 0 ) {
			$message[] = __( 'Tag(s) Added: ', 'wp-marketing-automations-crm' ) . array_reduce( $tags_added['assigned'], function ( $carry, $item ) {
					$name = $item->get_name();
					if ( empty( $carry ) ) {
						return $name;
					}

					return ! empty( $name ) ? $carry . ', ' . $name : $carry;
				}, '' );
		}

		return array(
			'status'  => self::$RESPONSE_SUCCESS,
			'message' => $message,
		);
	}
}

/**
 * Register action
 */
BWFCRM_Core()->actions->register_action( 'add_tags', 'BWFCRM\Actions\Autonami\Add_Tags', 'Add Tags', 'autonami', 'FunnelKit Automations' );
