<?php

if ( class_exists( 'WooCommerce' ) ) {
	class BWFAN_Rule_Customer_Order_Count extends BWFAN_Rule_Base {

		public $supports = array( 'order' );

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_order_count' );
		}

		/** v2 Methods: START */

		public function get_rule_type() {
			return 'Number';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( false, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( false, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );
			}

			$count = ( $customer instanceof WooFunnels_Customer ) ? absint( $customer->get_total_order_count() ) : 0;
			$value = absint( $rule_data['data'] );

			switch ( $rule_data['rule'] ) {
				case '==':
					$result = $count === $value;
					break;
				case '!=':
					$result = $count !== $value;
					break;
				case '>':
					$result = $count > $value;
					break;
				case '<':
					$result = $count < $value;
					break;
				case '>=':
					$result = $count >= $value;
					break;
				case '<=':
					$result = $count <= $value;
					break;
				default:
					$result = false;
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_condition_input_type() {
			return 'Text';
		}

		public function is_match( $rule_data ) {
			/**
			 * @var Woofunnels_Customer $customer
			 */
			$customer       = BWFAN_Core()->rules->getRulesData( 'bwf_customer' );
			$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
			$user_id        = BWFAN_Core()->rules->getRulesData( 'user_id' );

			$contact_id = BWFAN_Core()->rules->getRulesData( 'contact_id' );
			if ( ! empty( $contact_id ) && ! $customer instanceof WooFunnels_Customer ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = BWFAN_Core()->rules->getRulesData( 'email' );
				if ( ! is_email( $email ) ) {
					$order = BWFAN_Core()->rules->getRulesData( 'wc_order' );
					$email = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( false, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );
			}

			$count = $customer instanceof WooFunnels_Customer ? absint( $customer->get_total_order_count() ) : 0;
			$value = absint( $rule_data['condition'] );

			switch ( $rule_data['operator'] ) {
				case '==':
					$result = $count === $value;
					break;
				case '!=':
					$result = $count !== $value;
					break;
				case '>':
					$result = $count > $value;
					break;
				case '<':
					$result = $count < $value;
					break;
				case '>=':
					$result = $count >= $value;
					break;
				case '<=':
					$result = $count <= $value;
					break;
				default:
					$result = false;
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		public function ui_view() {
			esc_html_e( 'Contact Orders Count', 'autonami-automations-pro' );
			?>

            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>'); %>

            <%= ops[operator] %>
            <%= condition %>
			<?php
		}

		public function get_possible_rule_operators() {
			return array(
				'==' => __( 'is equal to', 'wp-marketing-automations' ),
				'!=' => __( 'is not equal to', 'wp-marketing-automations' ),
				'>'  => __( 'is greater than', 'wp-marketing-automations' ),
				'<'  => __( 'is less than', 'wp-marketing-automations' ),
				'>=' => __( 'is greater or equal to', 'wp-marketing-automations' ),
				'<=' => __( 'is less or equal to', 'wp-marketing-automations' ),
			);
		}

	}

	class BWFAN_Rule_Customer_Total_Spent extends BWFAN_Rule_Base {
		public $supports = array( 'order' );

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_total_spent' );
		}

		public function get_condition_input_type() {
			return 'Text';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( false, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( false, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );
			}

			$count = ( $customer instanceof WooFunnels_Customer ) ? (float) $customer->get_total_order_value() : 0;
			$value = (float) $rule_data['data'];

			switch ( $rule_data['rule'] ) {
				case '==':
					$result = $count === $value;
					break;
				case '!=':
					$result = $count !== $value;
					break;
				case '>':
					$result = $count > $value;
					break;
				case '<':
					$result = $count < $value;
					break;
				case '>=':
					$result = $count >= $value;
					break;
				case '<=':
					$result = $count <= $value;
					break;
				default:
					$result = false;
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		public function is_match( $rule_data ) {
			/**
			 * @var Woofunnels_Customer $customer
			 */
			$customer       = BWFAN_Core()->rules->getRulesData( 'bwf_customer' );
			$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
			$user_id        = BWFAN_Core()->rules->getRulesData( 'user_id' );

			$contact_id = BWFAN_Core()->rules->getRulesData( 'contact_id' );
			if ( ! empty( $contact_id ) && ! $customer instanceof WooFunnels_Customer ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = BWFAN_Core()->rules->getRulesData( 'email' );
				if ( ! is_email( $email ) ) {
					$order = BWFAN_Core()->rules->getRulesData( 'wc_order' );
					$email = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( false, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );
			}

			$count = $customer instanceof WooFunnels_Customer ? (float) $customer->get_total_order_value() : (float) 0;
			$value = (float) $rule_data['condition'];

			switch ( $rule_data['operator'] ) {
				case '==':
					$result = $count === $value;
					break;
				case '!=':
					$result = $count !== $value;
					break;
				case '>':
					$result = $count > $value;
					break;
				case '<':
					$result = $count < $value;
					break;
				case '>=':
					$result = $count >= $value;
					break;
				case '<=':
					$result = $count <= $value;
					break;
				default:
					$result = false;
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		public function ui_view() {
			esc_html_e( 'Contact Total Revenue', 'autonami-automations-pro' );
			?>

            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>'); %>

            <%= ops[operator] %>
            <%= condition %>
			<?php
		}

		public function get_possible_rule_operators() {
			return array(
				'==' => __( 'is equal to', 'wp-marketing-automations' ),
				'!=' => __( 'is not equal to', 'wp-marketing-automations' ),
				'>'  => __( 'is greater than', 'wp-marketing-automations' ),
				'<'  => __( 'is less than', 'wp-marketing-automations' ),
				'>=' => __( 'is greater or equal to', 'wp-marketing-automations' ),
				'<=' => __( 'is less or equal to', 'wp-marketing-automations' ),
			);
		}

	}

	class BWFAN_Rule_Customer_Purchased_Products extends BWFAN_Rule_Products {

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_purchased_products' );
		}

		/** v2 Methods: START */

		public function get_options( $term = '' ) {
			return $this->get_search_results( $term, true );
		}

		public function get_rule_type() {
			return 'Search';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( false, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( false, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );
				if ( ! $customer instanceof WooFunnels_Customer && 'none' === $rule_data['rule'] ) {
					return $this->return_is_match( true, $rule_data );
				}

				if ( ! $customer instanceof WooFunnels_Customer ) {
					return $this->return_is_match( false, $rule_data );
				}
			}

			$products = $customer->get_purchased_products();
			$data     = $rule_data['data'];

			$selected_products = array_map( function ( $product ) {
				return absint( $product['key'] );
			}, $data );
			$result            = $this->validate_set( $selected_products, $products, $rule_data['rule'] );

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function is_match( $rule_data ) {
			/**
			 * @var Woofunnels_Customer $customer
			 */
			$customer       = BWFAN_Core()->rules->getRulesData( 'bwf_customer' );
			$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
			$user_id        = BWFAN_Core()->rules->getRulesData( 'user_id' );

			$contact_id = BWFAN_Core()->rules->getRulesData( 'contact_id' );
			if ( ! empty( $contact_id ) && ! $customer instanceof WooFunnels_Customer ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = BWFAN_Core()->rules->getRulesData( 'email' );
				if ( ! is_email( $email ) ) {
					$order = BWFAN_Core()->rules->getRulesData( 'wc_order' );
					$email = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( false, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					return $this->return_is_match( false, $rule_data );
				}
			}

			$products      = $customer->get_purchased_products();
			$rule_products = array_map( 'absint', $rule_data['condition'] );
			$result        = $this->validate_set( $rule_products, $products, $rule_data['operator'] );

			return $this->return_is_match( $result, $rule_data );
		}

		public function validate_set( $products, $found_products, $operator ) {
			switch ( $operator ) {
				case 'any':
					$result = count( array_intersect( $products, $found_products ) ) > 0;
					break;
				case 'all':
					$result = count( array_intersect( $products, $found_products ) ) === count( $products );
					break;
				case 'none':
					$result = count( array_intersect( $products, $found_products ) ) === 0;
					break;

				default:
					$result = false;
					break;
			}

			return $result;
		}

		public function ui_view() {
			esc_html_e( 'Contact Purchased Products', 'autonami-automations-pro' );
			?>
            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>');%>

            <%= ops[operator] %> <% var chosen = []; %>
            <% _.each(condition, function( value, key ){ %>
            <% chosen.push(uiData[value]); %>

            <% }); %>
            <%= chosen.join(" | ") %>
			<?php
		}

	}

	class BWFAN_Rule_Customer_Purchased_Cat extends BWFAN_Rule_Term_Taxonomy {

		public $taxonomy_name = 'product_cat';

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_purchased_cat' );
		}

		/** v2 Methods: START */

		public function get_options( $term = '' ) {
			return $this->get_search_results( $term, true );
		}

		public function get_rule_type() {
			return 'Search';
		}

		/** v2 Methods: END */

		public function get_term_ids( $automation_data = [] ) {

			if ( isset( $automation_data['global'] ) && is_array( $automation_data['global'] ) ) {
				$customer   = '';
				$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
				if ( ! empty( $contact_id ) ) {
					$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
					$customer = new WooFunnels_Customer( $contact );
				}
				$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
				$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;
				$email          = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : '';
				$order_id       = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
				$order          = wc_get_order( $order_id );

			} else {

				/**
				 * @var Woofunnels_Customer $customer
				 */
				$customer       = BWFAN_Core()->rules->getRulesData( 'bwf_customer' );
				$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
				$user_id        = BWFAN_Core()->rules->getRulesData( 'user_id' );
				$email          = BWFAN_Core()->rules->getRulesData( 'email' );
				$order          = BWFAN_Core()->rules->getRulesData( 'wc_order' );

			}

			if ( ! $customer instanceof WooFunnels_Customer ) {
				if ( ! is_email( $email ) ) {
					$email = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return array();
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					return array();
				}
			}

			return $customer->get_purchased_products_cats();
		}

		public function get_possible_rule_operators() {
			return array(
				'any'  => __( 'matches any of', 'wp-marketing-automations' ),
				'all'  => __( 'matches all of ', 'wp-marketing-automations' ),
				'none' => __( 'matches none of ', 'wp-marketing-automations' ),
			);
		}

		public function ui_view() {
			esc_html_e( 'Contact Purchased Products Category', 'autonami-automations-pro' );
			?>
            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>'); %>

            <%= ops[operator] %><% var chosen = []; %>
            <% _.each(condition, function( value, key ){ %>
            <% chosen.push(uiData[value]); %>

            <% }); %>
            <%= chosen.join("/") %>
			<?php
		}

	}

	class BWFAN_Rule_Customer_Purchased_Tags extends BWFAN_Rule_Term_Taxonomy {

		public $taxonomy_name = 'product_tag';

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_purchased_tags' );
		}

		/** v2 Methods: START */

		public function get_options( $term = '' ) {
			return $this->get_search_results( $term, true );
		}

		public function get_rule_type() {
			return 'Search';
		}

		/** v2 Methods: END */

		public function get_term_ids( $automation_data = [] ) {
			/**
			 * @var Woofunnels_Customer $customer
			 */
			$contact_id     = BWFAN_Core()->rules->getRulesData( 'contact_id' );
			$customer       = BWFAN_Core()->rules->getRulesData( 'bwf_customer' );
			$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
			$user_id        = BWFAN_Core()->rules->getRulesData( 'user_id' );
			$email          = BWFAN_Core()->rules->getRulesData( 'email' );
			$order          = BWFAN_Core()->rules->getRulesData( 'wc_order' );

			if ( isset( $automation_data['global'] ) && is_array( $automation_data['global'] ) ) {
				$customer   = '';
				$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
				if ( ! empty( $contact_id ) ) {
					$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
					$customer = new WooFunnels_Customer( $contact );
				}
				$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
				$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;
				$email          = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : '';
				$order_id       = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
				$order          = wc_get_order( $order_id );
			}

			if ( $customer instanceof WooFunnels_Customer && $customer->get_id() > 0 ) {
				return $customer->get_purchased_products_tags();
			}


			if ( ! empty( $contact_id ) ) {
				$contact = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				if ( $contact instanceof WooFunnels_Contact && $contact->get_id() > 0 ) {
					$customer = new WooFunnels_Customer( $contact );

					return $customer->get_purchased_products_tags();
				}
			}

			if ( ! is_email( $email ) && ! empty( $user_id ) ) {
				$user_data = get_userdata( $user_id );
				$email     = $user_data instanceof WP_User ? $user_data->user_email : '';
			}

			/** get email from abandoned data */
			if ( ! is_email( $email ) ) {
				$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
			}

			if ( ! is_email( $email ) && bwfan_is_woocommerce_active() ) {
				$email = $order instanceof WC_Order ? $order->get_billing_email() : false;
			}

			if ( ! is_email( $email ) ) {
				return array();
			}

			$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

			if ( ! $customer instanceof WooFunnels_Customer ) {
				return array();
			}

			return $customer->get_purchased_products_tags();
		}

		public function ui_view() {
			esc_html_e( 'Contact Purchased Products Tag', 'autonami-automations-pro' );
			?>
            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>'); %>

            <%= ops[operator] %><% var chosen = []; %>
            <% _.each(condition, function( value, key ){ %>
            <% chosen.push(uiData[value]); %>

            <% }); %>
            <%= chosen.join("/") %>
			<?php
		}

	}

	class BWFAN_Rule_Customer_Country extends BWFAN_Rule_Country {

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_country' );
		}

		public function get_objects_country( $automation_data = [] ) {

			if ( isset( $automation_data['global'] ) && is_array( $automation_data['global'] ) ) {
				$customer   = '';
				$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
				if ( ! empty( $contact_id ) ) {
					$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
					$customer = new WooFunnels_Customer( $contact );
				}
				$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
				$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;
				$email          = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : '';
				$order_id       = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
				$order          = wc_get_order( $order_id );

			} else {

				/**
				 * @var Woofunnels_Customer $customer
				 */
				$contact_id     = BWFAN_Core()->rules->getRulesData( 'contact_id' );
				$customer       = BWFAN_Core()->rules->getRulesData( 'bwf_customer' );
				$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
				$user_id        = BWFAN_Core()->rules->getRulesData( 'user_id' );
				$email          = BWFAN_Core()->rules->getRulesData( 'email' );
				$order          = BWFAN_Core()->rules->getRulesData( 'wc_order' );

			}

			$contact = $customer instanceof WooFunnels_Customer ? $customer->contact : false;

			if ( ! empty( $contact_id ) && ! $contact instanceof WooFunnels_Contact ) {
				$contact = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
			}

			if ( ! is_email( $email ) && ! empty( $user_id ) ) {
				$user_data = get_userdata( $user_id );
				$email     = $user_data instanceof WP_User ? $user_data->user_email : '';
			}

			/** get email from abandoned data */
			if ( ! is_email( $email ) ) {
				$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
			}

			if ( ! $contact instanceof WooFunnels_Contact ) {
				if ( ! is_email( $email ) ) {
					$email = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				if ( ! is_email( $email ) ) {
					return false;
				}

				$contact = new WooFunnels_Contact( '', $email );

				if ( ! $contact instanceof WooFunnels_Contact ) {
					return false;
				}
			}

			$country = $contact->get_country();
			if ( empty( $country ) ) {
				return false;
			}

			return array( $country );
		}

		public function ui_view() {
			esc_html_e( 'Contact Country', 'autonami-automations-pro' );
			?>
            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>'); %>

            <%= ops[operator] %>
            <% var chosen = []; %>
            <% _.each(condition, function( value, key ){ %>
            <% chosen.push(uiData[value]); %>

            <% }); %>
            <%= chosen.join("/") %>
			<?php
		}

	}

	class BWFAN_Rule_Customer_Past_Purchased_Products extends BWFAN_Rule_Products {

		public function __construct() {
			$this->v2 = true;
			parent::__construct( 'customer_past_purchased_products' );
		}

		/** v2 Methods: START */

		public function get_options( $term = '' ) {
			return $this->get_search_results( $term, true );
		}

		public function get_rule_type() {
			return 'Search';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( false, $rule_data );
			}

			$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;

			$order = wc_get_order( $order_id );

			$order_created_date = '';
			$email              = '';
			if ( $order instanceof WC_Order ) {
				$order_created_date = $order->get_date_created();
				$email              = $order->get_billing_email();
			}

			/** Get email from abandoned cart data */
			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			if ( ! is_email( $email ) && ! empty( $abandoned_data ) && is_array( $abandoned_data ) ) {
				$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
			}

			if ( ! is_email( $email ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : '';
			}

			/** Get email from WooFunnels contact */
			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			if ( ! is_email( $email ) && ! empty( $contact_id ) ) {
				$contact = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$email   = $contact->get_email();
			}

			/** Get email from the user id */
			$user_id = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;
			if ( ! empty( $user_id ) && ! is_email( $email ) ) {
				$user_data = get_userdata( $user_id );
				$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
			}

			if ( ! is_email( $email ) ) {
				return $this->return_is_match( false, $rule_data );
			}

			$data              = $rule_data['data'];
			$selected_products = array_map( function ( $product ) {
				return absint( $product['key'] );
			}, $data );

			$current_time = ! empty( $order_created_date ) ? $order_created_date->date( 'Y-m-d H:i:s' ) : current_time( 'mysql' );

			/** Fetch products purchased before given time */
			$products = BWFAN_PRO_Common::get_customer_past_purchased_product( $current_time, $email );

			$result = $this->validate_set( $selected_products, $products, $rule_data['rule'] );

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function is_match( $rule_data ) {

			/** @var WC_Order $order */
			$order = BWFAN_Core()->rules->getRulesData( 'wc_order' );

			$order_created_date = '';
			$email              = '';
			if ( $order instanceof WC_Order ) {
				$order_created_date = $order->get_date_created();
				$email              = $order->get_billing_email();
			}

			/** Get email from abandoned cart data */
			$abandoned_data = BWFAN_Core()->rules->getRulesData( 'abandoned_data' );
			if ( ! is_email( $email ) && ! empty( $abandoned_data ) && is_array( $abandoned_data ) ) {
				$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
			}

			if ( ! is_email( $email ) ) {
				$email = BWFAN_Core()->rules->getRulesData( 'email' );
			}

			/** Get email from WooFunnels contact */
			$contact_id = BWFAN_Core()->rules->getRulesData( 'contact_id' );
			if ( ! is_email( $email ) && ! empty( $contact_id ) ) {
				$contact = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$email   = $contact->get_email();
			}

			/** Get email from the user id */
			$user_id = BWFAN_Core()->rules->getRulesData( 'user_id' );
			if ( ! empty( $user_id ) && ! is_email( $email ) ) {
				$user_data = get_userdata( $user_id );
				$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
			}

			if ( ! is_email( $email ) ) {
				return $this->return_is_match( false, $rule_data );
			}

			$rule_products = array_map( 'absint', $rule_data['condition'] );
			$current_time  = ! empty( $order_created_date ) ? $order_created_date->date( 'Y-m-d H:i:s' ) : current_time( 'mysql' );

			/** Fetch products purchased before given time */
			$products = BWFAN_PRO_Common::get_customer_past_purchased_product( $current_time, $email );

			$result = $this->validate_set( $rule_products, $products, $rule_data['operator'] );

			return $this->return_is_match( $result, $rule_data );
		}

		/**
		 * Perform rule validations
		 *
		 * @param $products
		 * @param $found_products
		 * @param $operator
		 *
		 * @return bool
		 */
		public function validate_set( $products, $found_products, $operator ) {
			switch ( $operator ) {
				case 'any':
					$result = count( array_intersect( $products, $found_products ) ) > 0;
					break;
				case 'all':
					$result = count( array_intersect( $products, $found_products ) ) === count( $products );
					break;
				case 'none':
					$result = count( array_intersect( $products, $found_products ) ) === 0;
					break;
				default:
					$result = false;
					break;
			}

			return $result;
		}

		public function ui_view() {
			esc_html_e( 'Past Purchased Products', 'autonami-automations-pro' );
			?>
            <% var ops = JSON.parse('<?php echo wp_json_encode( $this->get_possible_rule_operators() ); ?>');%>

            <%= ops[operator] %> <% var chosen = []; %>
            <% _.each(condition, function( value, key ){ %>
            <% chosen.push(uiData[value]); %>

            <% }); %>
            <%= chosen.join(" | ") %>
			<?php
		}

	}

	class BWFAN_Rule_Customer_Has_Purchased extends BWFAN_Rule_Base {
		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_has_purchased' );
		}

		public function get_possible_rule_values() {
			$operators = array(
				'yes' => __( 'Yes', 'wp-marketing-automations' ),
				'no'  => __( 'No', 'wp-marketing-automations' ),
			);

			return $operators;
		}

		/** v2 Methods: START */
		public function get_options( $term = '' ) {
			return $this->get_possible_rule_values();
		}

		public function get_rule_type() {
			return 'Select';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					$result = ( 'no' === $rule_data['data'] ) ? true : $result;

					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					$result = ( 'no' === $rule_data['data'] ) ? true : $result;

					return $this->return_is_match( $result, $rule_data );
				}
			}

			$count = $customer->get_total_order_count() ? absint( $customer->get_total_order_count() ) : 0;
			if ( ! empty( $count ) && $count > 0 ) {
				$result = true;
			}

			return $this->return_is_match( ( 'yes' === $rule_data['data'] ) ? $result : ! $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return null;
		}

	}

	class BWFAN_Rule_Customer_Has_Used_Coupons extends BWFAN_Rule_Base {
		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_has_used_coupons' );
		}

		public function get_possible_rule_values() {
			$operators = array(
				'yes' => __( 'Yes', 'wp-marketing-automations' ),
				'no'  => __( 'No', 'wp-marketing-automations' ),
			);

			return $operators;
		}

		/** v2 Methods: START */
		public function get_options( $term = '' ) {
			return $this->get_possible_rule_values();
		}

		public function get_rule_type() {
			return 'Select';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					$result = ( 'no' === $rule_data['data'] ) ? true : $result;

					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					$result = ( 'no' === $rule_data['data'] ) ? true : $result;

					return $this->return_is_match( $result, $rule_data );
				}
			}

			$count = $customer->get_used_coupons() ? count( $customer->get_used_coupons() ) : 0;
			if ( ! empty( $count ) && $count > 0 ) {
				$result = true;
			}

			return $this->return_is_match( ( 'yes' === $rule_data['data'] ) ? $result : ! $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return null;
		}

	}

	class BWFAN_Rule_Customer_First_Order_Date extends BWFAN_Rule_Base {
		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_first_order_date' );
		}

		/** v2 Methods: START */
		public function get_rule_type() {
			return 'Date';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					return $this->return_is_match( $result, $rule_data );
				}
			}

			$first_order_dt = $customer->get_f_order_date() ? date( 'Y-m-d', strtotime( $customer->get_f_order_date() ) ) : '';
			if ( empty( $first_order_dt ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$type           = $rule_data['rule'];
			$first_order_dt = strtotime( $first_order_dt );

			if ( 'between' === $type && is_array( $rule_data['data'] ) ) {
				$from   = strtotime( $rule_data['data']['from'] );
				$to     = strtotime( $rule_data['data']['to'] );
				$result = ( ( $first_order_dt >= $from ) && ( $first_order_dt <= $to ) );

				return $this->return_is_match( $result, $rule_data );
			}

			$filter_value = strtotime( $rule_data['data'] );
			switch ( $type ) {
				case 'before':
					$result = ( $first_order_dt < $filter_value );
					break;
				case 'after':
					$result = ( $first_order_dt > $filter_value );
					break;
				case 'is':
					$result = ( $first_order_dt === $filter_value );
					break;
				default:
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return array(
				'before'  => __( 'Before', 'wp-marketing-automations' ),
				'after'   => __( 'After', 'wp-marketing-automations' ),
				'is'      => __( 'On', 'wp-marketing-automations' ),
				'between' => __( 'Between', 'wp-marketing-automations' ),
			);
		}

	}

	class BWFAN_Rule_Customer_Last_Order_Date extends BWFAN_Rule_Base {
		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_last_order_date' );
		}

		/** v2 Methods: START */
		public function get_rule_type() {
			return 'Date';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					return $this->return_is_match( $result, $rule_data );
				}
			}

			$last_order_dt = $customer->get_l_order_date() ? date( 'Y-m-d', strtotime( $customer->get_l_order_date() ) ) : '';
			if ( empty( $last_order_dt ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$type          = $rule_data['rule'];
			$last_order_dt = strtotime( $last_order_dt );

			if ( 'between' === $type && is_array( $rule_data['data'] ) ) {
				$from   = strtotime( $rule_data['data']['from'] );
				$to     = strtotime( $rule_data['data']['to'] );
				$result = ( ( $last_order_dt >= $from ) && ( $last_order_dt <= $to ) );

				return $this->return_is_match( $result, $rule_data );
			}

			$filter_value = strtotime( $rule_data['data'] );

			switch ( $type ) {
				case 'before':
					$result = ( $last_order_dt < $filter_value );
					break;
				case 'after':
					$result = ( $last_order_dt > $filter_value );
					break;
				case 'is':
					$result = ( $last_order_dt === $filter_value );
					break;
				default:
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return array(
				'before'  => __( 'Before', 'wp-marketing-automations' ),
				'after'   => __( 'After', 'wp-marketing-automations' ),
				'is'      => __( 'On', 'wp-marketing-automations' ),
				'between' => __( 'Between', 'wp-marketing-automations' ),
			);
		}

	}

	class BWFAN_Rule_Customer_Last_Order_Days extends BWFAN_Rule_Base {
		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_last_order_days' );
		}

		/** v2 Methods: START */
		public function get_rule_type() {
			return 'Days';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					return $this->return_is_match( $result, $rule_data );
				}
			}

			$last_order_dt = $customer->get_l_order_date() ? date( 'Y-m-d', strtotime( $customer->get_l_order_date() ) ) : '';//excluding time
			if ( empty( $last_order_dt ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$type          = $rule_data['rule'];
			$last_order_dt = strtotime( $last_order_dt );

			$result = $this->validate_matches_duration_set( $last_order_dt, $rule_data, $type );

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return array(
				'over'    => __( 'Before', 'wp-marketing-automations' ),
				'past'    => __( 'In the last', 'wp-marketing-automations' ),
				'between' => __( 'In between', 'wp-marketing-automations' ),
			);
		}

	}

	class BWFAN_Rule_Customer_Average_Order_Value extends BWFAN_Rule_Base {
		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_average_order_value' );
		}

		/** v2 Methods: START */
		public function get_rule_type() {
			return 'Number';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );
			}

			$aov = ( $customer instanceof WooFunnels_Customer ) ? $customer->get_aov() : 0;

			$type = $rule_data['rule'];
			$data = $rule_data['data'];

			switch ( $type ) {
				case '<':
					$result = ( $aov < $data );
					break;
				case '>':
					$result = ( $aov > $data );
					break;
				default:
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return array(
				'<' => __( 'Less Than', 'wp-marketing-automations' ),
				'>' => __( 'More Than', 'wp-marketing-automations' ),
			);
		}

	}

	class BWFAN_Rule_Customer_Used_Coupons extends BWFAN_Rule_Base {

		public function __construct() {
			$this->v1 = false;
			$this->v2 = true;
			parent::__construct( 'customer_used_coupons' );
		}

		/** v2 Methods: START */

		public function get_options( $term = '' ) {
			return $this->get_search_results( $term, true );
		}

		public function get_rule_type() {
			return 'Search';
		}

		public function is_match_v2( $automation_data, $rule_data ) {
			$result = false;
			if ( ! isset( $automation_data['global'] ) || ! is_array( $automation_data['global'] ) ) {
				return $this->return_is_match( $result, $rule_data );
			}

			$abandoned_data = isset( $automation_data['global']['abandoned_data'] ) ? $automation_data['global']['abandoned_data'] : [];
			$user_id        = isset( $automation_data['global']['user_id'] ) ? $automation_data['global']['user_id'] : 0;

			$contact_id = isset( $automation_data['global']['contact_id'] ) ? $automation_data['global']['contact_id'] : 0;
			$customer   = '';
			if ( ! empty( $contact_id ) ) {
				$contact  = new WooFunnels_Contact( '', '', '', absint( $contact_id ) );
				$customer = new WooFunnels_Customer( $contact );
			}

			if ( ! $customer instanceof WooFunnels_Customer || 0 === absint( $customer->get_id() ) ) {
				$email = isset( $automation_data['global']['email'] ) ? $automation_data['global']['email'] : 0;
				if ( ! is_email( $email ) ) {
					$order_id = isset( $automation_data['global']['order_id'] ) ? $automation_data['global']['order_id'] : 0;
					$order    = wc_get_order( $order_id );
					$email    = $order instanceof WC_Order ? $order->get_billing_email() : false;
				}

				/** get email from the user id */
				if ( ! empty( $user_id ) && ! is_email( $email ) ) {
					$user_data = get_userdata( $user_id );
					$email     = $user_data instanceof WP_User ? $user_data->user_email : false;
				}

				/** get email from abandoned data */
				if ( ! is_email( $email ) ) {
					$email = isset( $abandoned_data['email'] ) ? $abandoned_data['email'] : false;
				}

				if ( ! is_email( $email ) ) {
					$result = ( 'none' === $rule_data['rule'] ) ? true : $result;

					return $this->return_is_match( $result, $rule_data );
				}

				$customer = BWFAN_PRO_Common::get_bwf_customer_by_email( $email );

				if ( ! $customer instanceof WooFunnels_Customer ) {
					$result = ( 'none' === $rule_data['rule'] ) ? true : $result;

					return $this->return_is_match( $result, $rule_data );
				}
			}

			$selected_coupons = array_map( function ( $coupon ) {
				return strtolower( $coupon['label'] );
			}, $rule_data['data'] );
			$type             = $rule_data['rule'];

			/** If type contains and not_contains - selected coupons can be multiple checking directly using query */
			if ( 'contains' === $type || 'not_contains' === $type ) {
				$result = $this->coupon_contains_text( $customer->get_cid(), $selected_coupons, $type );

				return $this->return_is_match( $result, $rule_data );
			}

			$used_coupons = $customer->get_used_coupons() ? $customer->get_used_coupons() : '';

			if ( empty( $used_coupons ) ) {
				$result = ( 'none' === $rule_data['rule'] ) ? true : $result;

				return $this->return_is_match( $result, $rule_data );
			}

			$used_coupons = array_map( 'strtolower', $used_coupons );

			switch ( $type ) {
				case 'any':
					$result = count( array_intersect( $selected_coupons, $used_coupons ) ) > 0;
					break;
				case 'none':
					$result = count( array_intersect( $selected_coupons, $used_coupons ) ) === 0;
					break;
				default:
					break;
			}

			return $this->return_is_match( $result, $rule_data );
		}

		/** v2 Methods: END */

		public function get_possible_rule_operators() {
			return array(
				'any'          => __( 'Matches any of', 'wp-marketing-automations' ),
				'none'         => __( 'Matches none of', 'wp-marketing-automations' ),
				'contains'     => __( 'contains', 'wp-marketing-automations' ),
				'not_contains' => __( 'does not contains', 'wp-marketing-automations' ),
			);
		}

		public function get_search_results( $term, $v2 = false ) {
			$array = BWFAN_Common::get_coupon( $term );

			if ( $v2 ) {
				$return = array();
				foreach ( $array['results'] as $coupon ) {
					$return[ $coupon['id'] ] = $coupon['text'];
				}

				return $return;
			}
		}

		/**
		 * Check used coupons contains or not contains text
		 *
		 * @param $cid
		 * @param $values
		 * @param $type
		 *
		 * @return bool
		 */
		public function coupon_contains_text( $cid, $values, $type ) {
			global $wpdb;
			$operator     = 'contains' === $type ? 'LIKE' : 'NOT LIKE';
			$search_query = array_map( function ( $coupon ) use ( $operator ) {
				return '`used_coupons` ' . $operator . ' "%' . $coupon . '%"';
			}, $values );

			$search_query = implode( ' OR ', $search_query );

			$query = "SELECT id FROM {$wpdb->prefix}bwf_wc_customers WHERE cid = $cid AND $search_query";

			return ! empty ( $wpdb->get_var( $query ) );
		}
	}
}
