<?php
/**
 * Plugin Name: FlowMattic Integration WABot
 * Plugin URI: https://flowmattic.com/
 * Description: FlowMattic integration for unofficial WhatsApp API.
 * Version: 1.0
 * Requires at least: 5.0
 * Requires PHP:      7.2
 * Author: InfiWebs
 * Author URI: https://www.infiwebs.com
 * Textdomain: flowmattic
 *
 * @package FlowMattic
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin Root File.
if ( ! defined( 'WABOT_PLUGIN_FILE' ) ) {
	define( 'WABOT_PLUGIN_FILE', __FILE__ );
}

// Plugin Folder URL.
if ( ! defined( 'WABOT_PLUGIN_URL' ) ) {
	define( 'WABOT_PLUGIN_URL', plugin_dir_url( WABOT_PLUGIN_FILE ) );
}

/**
 * Wabot_Integration integration class.
 *
 * @since 1.0
 */
class FlowMattic_Wabot {
	/**
	 * Constructor.
	 *
	 * @access public
	 * @since 1.0
	 * @return void
	 */
	public function __construct() {
		// Enqueue custom view for wabot.
		add_action( 'flowmattic_enqueue_views', array( $this, 'enqueue_views' ) );

		flowmattic_add_application(
			'wabot',
			array(
				'name'         => esc_attr__( 'WABot Integration', 'flowmattic' ),
				'icon'         => WABOT_PLUGIN_URL . '/icon.png',
				'instructions' => __( 'Unofficial WhatsApp API Integration', 'flowmattic' ),
				'actions'      => $this->get_actions(),
				'type'         => 'action',
				'version'      => '1.0',
			)
		);
	}

	/**
	 * Enqueue view js.
	 *
	 * @access public
	 * @since 1.0
	 * @return void
	 */
	public function enqueue_views() {
		wp_enqueue_script( 'flowmattic-app-view-wabot', WABOT_PLUGIN_URL . '/view-wabot.js', array( 'flowmattic-workflow-utils' ), wp_rand(), true );

		// Load template file.
		include wp_normalize_path( __DIR__ . '/template.tpl' );
	}

	/**
	 * Set actions.
	 *
	 * @access public
	 * @since 1.0
	 * @return array
	 */
	public function get_actions() {
		return array(
			'send_text_message'   => array(
				'title' => esc_attr__( 'Send Text Message', 'flowmattic' ),
			),
			'send_media_message'  => array(
				'title' => esc_attr__( 'Send Media Message', 'flowmattic' ),
			),
			'send_text_to_group'  => array(
				'title' => esc_attr__( 'Send Text to Group', 'flowmattic' ),
			),
			'send_media_to_group' => array(
				'title' => esc_attr__( 'Send Media to Group', 'flowmattic' ),
			),
			'send_button_json'    => array(
				'title' => esc_attr__( 'Send Buttons', 'flowmattic' ),
			),
		);
	}

	/**
	 * Run the action step.
	 *
	 * @access public
	 * @since 1.0
	 * @param string $workflow_id  Workflow ID.
	 * @param object $step         Workflow current step.
	 * @param array  $capture_data Data captured by the WordPress action.
	 * @return array
	 */
	public function run_action_step( $workflow_id, $step, $capture_data ) {

		$action = $step['action'];
		$fields = isset( $step['fields'] ) ? $step['fields'] : ( isset( $step['actionAppArgs'] ) ? $step['actionAppArgs'] : array() );

		switch ( $action ) {
			case 'send_text_message':
				$response = $this->send_text_message( $fields );
				break;

			case 'send_media_message':
				$response = $this->send_media_message( $fields );
				break;

			case 'send_text_to_group':
				$response = $this->send_text_to_group( $fields );
				break;

			case 'send_media_to_group':
				$response = $this->send_media_to_group( $fields );
				break;

			case 'send_button_json':
				$response = $this->send_button_json( $fields );
				break;
		}

		return $response;
	}

	/**
	 * Send text message.
	 *
	 * @access public
	 * @since 1.0
	 * @param array $fields Workflow data fields.
	 * @return array
	 */
	public function send_text_message( $fields ) {
		$subdomain    = $fields['subdomain'];
		$number       = $fields['number'];
		$message      = rawurlencode( $fields['message'] );
		$instance_id  = $fields['instance_id'];
		$access_token = $fields['access_token'];

		// @codingStandardsIgnoreStart
		$curl = curl_init();

		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL            => 'https://' . $subdomain . '.wabot.my/api/send.php?number=' . $number . '&type=text&message=' . $message . '&instance_id=' . $instance_id . '&access_token=' . $access_token,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_HTTPHEADER     => array(
					'Content-Type: application/json',
				),
			)
		);

		$response = curl_exec( $curl );

		curl_close( $curl );

		// @codingStandardsIgnoreEnd

		$response       = json_decode( $response, true );
		$response_array = array();

		foreach ( $response as $key => $value ) {
			if ( is_array( $value ) ) {
				$response_array = flowmattic_recursive_array( $response_array, $key, $value );
			} else {
				$response_array[ $key ] = $value;
			}
		}

		return wp_json_encode( $response_array );
	}

	/**
	 * Send media message.
	 *
	 * @access public
	 * @since 1.0
	 * @param array $fields Workflow data fields.
	 * @return array
	 */
	public function send_media_message( $fields ) {
		$subdomain    = $fields['subdomain'];
		$number       = $fields['number'];
		$message      = rawurlencode( $fields['message'] );
		$media_url    = $fields['media_url'];
		$instance_id  = $fields['instance_id'];
		$access_token = $fields['access_token'];

		// @codingStandardsIgnoreStart
		$curl = curl_init();

		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL            => 'https://' . $subdomain . '.wabot.my/api/send.php?number=' . $number . '&type=media&message=' . $message . '&media_url=' . $media_url . '&filename=' . basename( $media_url ) . '&instance_id=' . $instance_id . '&access_token=' . $access_token,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_HTTPHEADER     => array(
					'Content-Type: application/json',
				),
			)
		);

		$response = curl_exec( $curl );

		curl_close( $curl );

		// @codingStandardsIgnoreEnd

		$response       = json_decode( $response, true );
		$response_array = array();

		foreach ( $response as $key => $value ) {
			if ( is_array( $value ) ) {
				$response_array = flowmattic_recursive_array( $response_array, $key, $value );
			} else {
				$response_array[ $key ] = $value;
			}
		}

		return wp_json_encode( $response_array );
	}

	/**
	 * Send text to group.
	 *
	 * @access public
	 * @since 1.0
	 * @param array $fields Workflow data fields.
	 * @return array
	 */
	public function send_text_to_group( $fields ) {
		$subdomain    = $fields['subdomain'];
		$number       = $fields['number'];
		$message      = rawurlencode( $fields['message'] );
		$instance_id  = $fields['instance_id'];
		$access_token = $fields['access_token'];
		$group_id     = $fields['group_id'];

		// @codingStandardsIgnoreStart
		$curl = curl_init();

		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL            => 'https://' . $subdomain . '.wabot.my/api/sendgroupmsg.php?number=' . $number . '&type=text&message=' . $message . '&group_id=' . $group_id . '&instance_id=' . $instance_id . '&access_token=' . $access_token,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_HTTPHEADER     => array(
					'Content-Type: application/json',
				),
			)
		);

		$response = curl_exec( $curl );

		curl_close( $curl );

		// @codingStandardsIgnoreEnd

		$response       = json_decode( $response, true );
		$response_array = array();

		foreach ( $response as $key => $value ) {
			if ( is_array( $value ) ) {
				$response_array = flowmattic_recursive_array( $response_array, $key, $value );
			} else {
				$response_array[ $key ] = $value;
			}
		}

		return wp_json_encode( $response_array );
	}

	/**
	 * Send media to group.
	 *
	 * @access public
	 * @since 1.0
	 * @param array $fields Workflow data fields.
	 * @return array
	 */
	public function send_media_to_group( $fields ) {
		$subdomain    = $fields['subdomain'];
		$number       = $fields['number'];
		$message      = rawurlencode( $fields['message'] );
		$media_url    = $fields['media_url'];
		$instance_id  = $fields['instance_id'];
		$access_token = $fields['access_token'];
		$group_id     = $fields['group_id'];

		// @codingStandardsIgnoreStart
		$curl = curl_init();

		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL            => 'https://' . $subdomain . '.wabot.my/api/sendgroupmsg.php?number=' . $number . '&type=media&message=' . $message . '&media_url=' . $media_url . '&filename=' . basename( $media_url ) . '&group_id=' . $group_id . '&instance_id=' . $instance_id . '&access_token=' . $access_token,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_HTTPHEADER     => array(
					'Content-Type: application/json',
				),
			)
		);

		$response = curl_exec( $curl );

		curl_close( $curl );

		// @codingStandardsIgnoreEnd

		$response       = json_decode( $response, true );
		$response_array = array();

		foreach ( $response as $key => $value ) {
			if ( is_array( $value ) ) {
				$response_array = flowmattic_recursive_array( $response_array, $key, $value );
			} else {
				$response_array[ $key ] = $value;
			}
		}

		return wp_json_encode( $response_array );
	}

	/**
	 * Send button json.
	 *
	 * @access public
	 * @since 1.0
	 * @param array $fields Workflow data fields.
	 * @return array
	 */
	public function send_button_json( $fields ) {
		$subdomain      = $fields['subdomain'];
		$number         = $fields['number'];
		$message        = $fields['message'];
		$footer_text    = $fields['footer_text'];
		$instance_id    = $fields['instance_id'];
		$access_token   = $fields['access_token'];
		$button_1_id    = $fields['button_1_id'];
		$button_1_title = $fields['button_1_title'];
		$button_2_id    = $fields['button_2_id'];
		$button_2_title = $fields['button_2_title'];
		$button_3_id    = $fields['button_3_id'];
		$button_3_title = $fields['button_3_title'];
		$button_json    = '';
		$buttons_array  = array();

		if ( '' !== $button_1_title ) {
			$buttons_array[] = array(
				'buttonId'   => $button_1_id,
				'buttonText' => array(
					'displayText' => $button_1_title,
				),
			);
		}
		if ( '' !== $button_2_title ) {
			$buttons_array[] = array(
				'buttonId'   => $button_2_id,
				'buttonText' => array(
					'displayText' => $button_2_title,
				),
			);
		}
		if ( '' !== $button_3_title ) {
			$buttons_array[] = array(
				'buttonId'   => $button_3_id,
				'buttonText' => array(
					'displayText' => $button_3_title,
				),
			);
		}

		$button_json = wp_json_encode(
			array(
				'text'    => $message,
				'footer'  => $footer_text,
				'buttons' => $buttons_array,
			)
		);

		// @codingStandardsIgnoreStart
		$curl = curl_init();

		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL            => 'https://' . $subdomain . '.wabot.my/api/send.php?number=' . $number . '&type=json&instance_id=' . $instance_id . '&access_token=' . $access_token,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_HTTPHEADER     => array(
					'Content-Type: application/json',
				),
				CURLOPT_POSTFIELDS     => $button_json,
			)
		);

		$response = curl_exec( $curl );

		curl_close( $curl );

		// @codingStandardsIgnoreEnd

		$response       = json_decode( $response, true );
		$response_array = array();

		foreach ( $response as $key => $value ) {
			if ( is_array( $value ) ) {
				$response_array = flowmattic_recursive_array( $response_array, $key, $value );
			} else {
				$response_array[ $key ] = $value;
			}
		}

		return wp_json_encode( $response_array );
	}

	/**
	 * Test action event ajax.
	 *
	 * @access public
	 * @since 1.0
	 * @param array $event_data Test event data.
	 * @return array
	 */
	public function test_event_action( $event_data ) {
		$event          = $event_data['event'];
		$fields         = isset( $event_data['fields'] ) ? $event_data['fields'] : ( isset( $settings['actionAppArgs'] ) ? $settings['actionAppArgs'] : array() );
		$workflow_id    = $event_data['workflow_id'];
		$response_array = array();

		// Replace action for testing.
		$event_data['action'] = $event;

		$request = $this->run_action_step( $workflow_id, $event_data, $fields );

		return $request;
	}
}

/**
 * Initialize the integration.
 *
 * @access public
 * @since 1.0
 * @return void
 */
function initialize_wabot_integration() {
	new FlowMattic_Wabot();
}

add_action( 'after_setup_theme', 'initialize_wabot_integration', 19 );
